/*=============================================================================
Copyright  2010 AMETEK Programmable Power. All rights reserved.

FILE NAME:  AmetekDC.c

PURPOSE:
   This IVI Driver sample program initializes the power supply, configures
   a channel to supply power, perfoms a few queries, and displays a message 
   pop-up dialog.

NOTES:
   IVI Instrument Driver for AMETEK Programmable Power XG Series Power Supply.
   The driver can be used to control RS-485 bus instruments via a RS-232, RS-232,
   GPIB, USB or Ethernet equipped unit.

   Developed using LabWindows/CVI 9.1, 2009
   
=============================================================================*/


/*****************************************************************************
 *---------------------------- Include Files --------------------------------*
 *****************************************************************************/
#include <string.h>
#include <stdio.h>  
#include <ctype.h>
#include <stdlib.h>
#include <utility.h>    
#include "AmetekDC.h"

// This define includes the code that  deals with the bus addressing
// scheme that is idiosyncratic in the XG.
#define __AMETEKDC_SPECIFIC_ADDRESS_HANDLING__

/*****************************************************************************
 *--------------------- Power Supply Property Declarations -----------------------*
 *****************************************************************************/
// String used to format output parameters
#define AMETEKDC_REAL_PARAM_OUTP_FORMAT			" %.4f"

// String used to format output parameters
#define AMETEKDC_LONG_PARAM_OUTP_FORMAT 		" %li"

// String used to format output parameters
#define AMETEKDC_ULONG_PARAM_OUTP_FORMAT 		" %lu"

// String used to format output parameters
#define AMETEKDC_SHORT_PARAM_OUTP_FORMAT 		" %hi"

// String used to format output parameters
#define AMETEKDC_USHORT_PARAM_OUTP_FORMAT 		" %hu"

// String used to format output parameters
#define AMETEKDC_STRING_PARAM_OUTP_FORMAT		" %s"

// String used to format input parameters (responses from XTR)
#define AMETEKDC_REAL_PARAM_INP_FORMAT			"%lf%*[^\n]"

// String used to format input parameters
#define AMETEKDC_LONG_PARAM_INP_FORMAT 			"%li%*[^\n]"

// String used to format input parameters
#define AMETEKDC_ULONG_PARAM_INP_FORMAT 		"%lu%*[^\n]"

// String used to format input parameters
#define AMETEKDC_SHORT_PARAM_INP_FORMAT 		"%hi%*[^\n]"

// String used to format input parameters
#define AMETEKDC_USHORT_PARAM_INP_FORMAT 		"%hu%*[^\n]"

// String used to format input parameters
#define AMETEKDC_STRING_PARAM_INP_FORMAT		"%s%*[^\n]"

// String used to format input parameters for the ViBoolean type.
#define AMETEKDC_BOOL_PARAM_INP_FORMAT			"%hu%*[^\n]"


/*****************************************************************************
 *--------------------- Driver Property Declarations -----------------------*
 *****************************************************************************/

// The write time per byte for the XG to save a word of data.
#define AMETEKDC_NVROM_WR_DELAY_PER_WORD_SEC	0.010

// Number of words written when a user memory slot is saved.
#define AMETEKDC_WORDS_SAVED_DURING_SAVE          8

// Allow the XG time to complete the command which writes to EEPROM.
#define AMETEKDC_SAVE_DELAY_SEC (0.0062 + AMETEKDC_WORDS_SAVED_DURING_SAVE * AMETEKDC_NVROM_WR_DELAY_PER_WORD_SEC )

// Number of bytes written to EEPROM during reset.
#define AMETEKDC_WORDS_SAVED_DURING_RESET		(AMETEKDC_WORDS_SAVED_DURING_SAVE * AMETEKDC_NUMB_USER_SETTINGS)

// Allow the XG time to reset or else the next command will fail.
#define AMETEKDC_RESET_DELAY_SEC   (0.022 + AMETEKDC_WORDS_SAVED_DURING_RESET * AMETEKDC_NVROM_WR_DELAY_PER_WORD_SEC)

// The maximum amount of time that can be expected after the last
// character has been transmitted and when the handler has parsed
// and begun executing the handler. (Doesn't include the command
// execution time which varies from command to command).
#define AMETEKDC_MAINLOOP_LATENCY_SEC  			  0.010



/*****************************************************************************
 *--------------------- Hidden Attribute Declarations -----------------------*
 *****************************************************************************/

#define AMETEKDC_ATTR_OPC_TIMEOUT                 	( IVI_SPECIFIC_PRIVATE_ATTR_BASE + 1L )
                     
#define AMETEKDC_ATTR_VOLTAGE_RANGE_TABLE_ADDRESS 	( IVI_SPECIFIC_PRIVATE_ATTR_BASE + 2L )
                     
#define AMETEKDC_ATTR_CURRENT_RANGE_TABLE_ADDRESS 	( IVI_SPECIFIC_PRIVATE_ATTR_BASE + 3L )
                     
#define AMETEKDC_ATTR_MULTICHANNEL_MODEL          	( IVI_SPECIFIC_PRIVATE_ATTR_BASE + 4L )
#define AMETEKDC_ATTR_OVP_DISABLE_VALUE           	( IVI_SPECIFIC_PRIVATE_ATTR_BASE + 6L )
#define AMETEKDC_ATTR_OCP_RESTORE_VALUE       		( IVI_SPECIFIC_PRIVATE_ATTR_BASE + 7L )
#define AMETEKDC_ATTR_OVP_RANGE_TABLE_ADDRESS 		( IVI_SPECIFIC_PRIVATE_ATTR_BASE + 8L )

/*****************************************************************************
 *---------------------------- Useful Macros --------------------------------*
 *****************************************************************************/

/*----- 488.2 Event Status Register(ESR ) Bits ------------------------------*/

#define IEEE_488_2_ERROR_BITS    0x003C /* CME, EXE, DDE, QYE */

#define SETUP_INTERFACE_BAUD 	"Baudrate"
#define SETUP_MODEL_TAG         "Model"
#define SETUP_CHANNEL_TAG       "Channels"

#define GPIB_IF			1
#define RS232_IF		2
#define ENET_IF			3

static short int		user_if;

//********************* Output Prot. OVP Range Table *****************************
static IviRangeTableEntry outputProtOvpRangTable6Entries[] =
	{
		{0.5, 7.5, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputProtOvpRangTable6 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputProtOvpRangTable6Entries,
	};

static IviRangeTableEntry outputProtOvpRangTable8Entries[] =
	{
		{0.5, 9.999, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputProtOvpRangTable8 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputProtOvpRangTable8Entries,
	};

static IviRangeTableEntry outputProtOvpRangTable12Entries[] =
	{
		{1, 15, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputProtOvpRangTable12 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputProtOvpRangTable12Entries,
	};

static IviRangeTableEntry outputProtOvpRangTable20Entries[] =
	{
		{1, 24, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputProtOvpRangTable20 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputProtOvpRangTable20Entries,
	};

static IviRangeTableEntry outputProtOvpRangTable33Entries[] =
	{
		{2, 39, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputProtOvpRangTable33 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputProtOvpRangTable33Entries,
	};

static IviRangeTableEntry outputProtOvpRangTable40Entries[] =
	{
		{2, 44, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputProtOvpRangTable40 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputProtOvpRangTable40Entries,
	};

static IviRangeTableEntry outputProtOvpRangTable60Entries[] =
	{
		{3, 66, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputProtOvpRangTable60 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputProtOvpRangTable60Entries,
	};

static IviRangeTableEntry outputProtOvpRangTable80Entries[] =
	{
		{3, 95, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputProtOvpRangTable80 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputProtOvpRangTable80Entries,
	};

static IviRangeTableEntry outputProtOvpRangTable100Entries[] =
	{
		{3, 125, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputProtOvpRangTable100 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputProtOvpRangTable100Entries,
	};

static IviRangeTableEntry outputProtOvpRangTable150Entries[] =
	{
		{3, 180, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputProtOvpRangTable150 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputProtOvpRangTable150Entries,
	};

static IviRangeTableEntry outputProtOvpRangTable300Entries[] =
	{
		{5, 330, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputProtOvpRangTable300 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputProtOvpRangTable300Entries,
	};

static IviRangeTableEntry outputProtOvpRangTable600Entries[] =
	{
		{5, 660, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputProtOvpRangTable600 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputProtOvpRangTable600Entries,
	};
//********************* Output Prot. OVP Range Table *****************************

static IviRangeTableEntry attrDataRateRangeTableEntries[] =
	{
		{1200, 0, 0, "1200", 0},
		{2400, 0, 0, "2400", 0},
		{4800, 0, 0, "4800", 0},
		{9600, 0, 0, "9600", 0},
		{19200, 0, 0, "19200", 0},
		{38400, 0, 0, "38400", 0},
		{57600, 0, 0, "57600", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable attrDataRateRangeTable =
	{
		IVI_VAL_DISCRETE,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        attrDataRateRangeTableEntries,
	};

static IviRangeTableEntry attrConfigurableDelayRangeTableEntries[] =
	{
		{AMETEKDC_MIN_CMD_DELAY_SEC, 60, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable attrConfigurableDelayRangeTable =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        attrConfigurableDelayRangeTableEntries,
	};

static IviRangeTableEntry attrAutoSequenceDwellTimeRangeTableEntries[] =
	{
		{0, AMETEKDC_MAX_DWELL_TIME_SEC, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable attrAutoSequenceDwellTimeRangeTable =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        attrAutoSequenceDwellTimeRangeTableEntries,
	};

static IviRangeTableEntry scpiCmdStringRangeTableEntries[] =
	{
		{AMETEKDC_VAL_SYST_FPANEL, 0, 0, "SYST%s:FPAN", 0},
		{AMETEKDC_VAL_SYST_PROT_MASK, 0, 0, "SYST%s:PROT:MASK", 0},
		{AMETEKDC_VAL_SYST_PROT_LATCH, 0, 0, "SYST%s:PROT:LATC", 0},
		{AMETEKDC_VAL_SYST_ERROR, 0, 0, "SYST%s:ERR", 0},
		{AMETEKDC_VAL_SYST_ERROR_CODE, 0, 0, "SYST%s:ERR:CODE", 0},
		{AMETEKDC_VAL_SYST_ERROR_COUNT, 0, 0, "SYST%s:ERR:COUN", 0},
		{AMETEKDC_VAL_SYST_VERS, 0, 0, "SYST%s:VERS", 0},
		{AMETEKDC_VAL_SYST_SAVE, 0, 0, "SYST%s:SAVE", 0},
		{AMETEKDC_VAL_SYST_REC, 0, 0, "SYST%s:REC", 0},
		{AMETEKDC_VAL_SYST_RES, 0, 0, "SYST%s:RES", 0},
		{AMETEKDC_VAL_SYST_WAIT, 0, 0, "SYST%s:WAIT", 0},
		{AMETEKDC_VAL_SYST_TEST, 0, 0, "SYST%s:TEST", 0},
		{AMETEKDC_VAL_SYST_IDEN, 0, 0, "SYST%s:IDEN", 0},
		{AMETEKDC_VAL_SYST_TRG, 0, 0, "SYST%s:INIT", 0},
		{AMETEKDC_VAL_SYST_REMOTE_STATE, 0, 0, "SYST%s:REM:STATe", 0},
		{AMETEKDC_VAL_SYST_REMOTE_SOUR_VOLT, 0, 0, "SYST%s:REM:SOUR", 0},
		{AMETEKDC_VAL_SYST_REMOTE_SOUR_CURR, 0, 0, "SYST%s:REM:SOUR:CURR", 0},
		{AMETEKDC_VAL_SYST_COMM_ADR, 0, 0, "SYST%s:COMM:ADDR", 0},
		{AMETEKDC_VAL_SYST_COMM_VAP_LEVEL, 0, 0, "SYST%s:COMM:APR:LEV", 0},
		{AMETEKDC_VAL_SYST_COMM_CAP_LEVEL, 0, 0, "SYST%s:COMM:APR:LEV:CURR", 0},
		{AMETEKDC_VAL_SYST_HELP, 0, 0, "SYST%s:HELP", 0},
		{AMETEKDC_VAL_SYST_HELP_LEGACY, 0, 0, "SYST%s:HELP:LEG", 0},
		{AMETEKDC_VAL_SYST_HELP_SYNTAX, 0, 0, "SYST%s:HELP:SYNT", 0},
		{AMETEKDC_VAL_STAT_PRESET, 0, 0, "STAT%s:PRES", 0},
		{AMETEKDC_VAL_STAT_CLEAR, 0, 0, "STAT%s:CLE", 0},
		{AMETEKDC_VAL_STAT_QUEST_EVENT, 0, 0, "STAT%s:QUES", 0},
		{AMETEKDC_VAL_STAT_QUEST_ENABLE, 0, 0, "STAT%s:QUES:ENAB", 0},
		{AMETEKDC_VAL_STAT_QUEST_COND, 0, 0, "STAT%s:QUES:COND", 0},
		{AMETEKDC_VAL_STAT_QUEST_PTR, 0, 0, "STAT%s:QUES:PTR", 0},
		{AMETEKDC_VAL_STAT_QUEST_NTR, 0, 0, "STAT%s:QUES:NTR", 0},
		{AMETEKDC_VAL_STAT_QUEST_VOLT_EVENT, 0, 0, "STAT%s:QUES:VOLT", 0},
		{AMETEKDC_VAL_STAT_QUEST_VOLT_ENABLE, 0, 0, "STAT%s:QUES:VOLT:ENAB", 0},
		{AMETEKDC_VAL_STAT_QUEST_VOLT_COND, 0, 0, "STAT%s:QUES:VOLT:COND", 0},
		{AMETEKDC_VAL_STAT_QUEST_VOLT_PTR, 0, 0, "STAT%s:QUES:VOLT:PTR", 0},
		{AMETEKDC_VAL_STAT_QUEST_VOLT_NTR, 0, 0, "STAT%s:QUES:VOLT:NTR", 0},
		{AMETEKDC_VAL_STAT_QUEST_TEMP_EVENT, 0, 0, "STAT%s:QUES:TEMP", 0},
		{AMETEKDC_VAL_STAT_QUEST_TEMP_ENABLE, 0, 0, "STAT%s:QUES:TEMP:ENAB", 0},
		{AMETEKDC_VAL_STAT_QUEST_TEMP_COND, 0, 0, "STAT%s:QUES:TEMP:COND", 0},
		{AMETEKDC_VAL_STAT_QUEST_TEMP_PTR, 0, 0, "STAT%s:QUES:TEMP:PTR", 0},
		{AMETEKDC_VAL_STAT_QUEST_TEMP_NTR, 0, 0, "STAT%s:QUES:TEMP:NTR", 0},
		{AMETEKDC_VAL_STAT_OPER_EVENT, 0, 0, "STAT%s:OPER", 0},
		{AMETEKDC_VAL_STAT_OPER_ENABLE, 0, 0, "STAT%s:OPER:ENAB", 0},
		{AMETEKDC_VAL_STAT_OPER_COND, 0, 0, "STAT%s:OPER:COND", 0},
		{AMETEKDC_VAL_STAT_OPER_PTR, 0, 0, "STAT%s:OPER:PTR", 0},
		{AMETEKDC_VAL_STAT_OPER_NTR, 0, 0, "STAT%s:OPER:NTR", 0},
		{AMETEKDC_VAL_STAT_OPER_CSH_EVENT, 0, 0, "STAT%s:OPER:CSH", 0},
		{AMETEKDC_VAL_STAT_OPER_CSH_ENABLE, 0, 0, "STAT%s:OPER:CSH:ENAB", 0},
		{AMETEKDC_VAL_STAT_OPER_CSH_COND, 0, 0, "STAT%s:OPER:CSH:COND", 0},
		{AMETEKDC_VAL_STAT_OPER_CSH_PTR, 0, 0, "STAT%s:OPER:CSH:PTR", 0},
		{AMETEKDC_VAL_STAT_OPER_CSH_NTR, 0, 0, "STAT%s:OPER:CSH:NTR", 0},
		{AMETEKDC_VAL_STAT_OPER_SHUT_EVENT, 0, 0, "STAT%s:OPER:SHUT", 0},
		{AMETEKDC_VAL_STAT_OPER_SHUT_ENABLE, 0, 0, "STAT%s:OPER:SHUT:ENAB", 0},
		{AMETEKDC_VAL_STAT_OPER_SHUT_COND, 0, 0, "STAT%s:OPER:SHUT:COND", 0},
		{AMETEKDC_VAL_STAT_OPER_SHUT_PTR, 0, 0, "STAT%s:OPER:SHUT:PTR", 0},
		{AMETEKDC_VAL_STAT_OPER_SHUT_NTR, 0, 0, "STAT%s:OPER:SHUT:NTR", 0},
		{AMETEKDC_VAL_STAT_OPER_SHUT_PROT_EVENT, 0, 0, "STAT%s:OPER:SHUT:PROT", 0},
		{AMETEKDC_VAL_STAT_OPER_SHUT_PROT_ENABLE, 0, 0, "STAT%s:OPER:SHUT:PROT:ENAB", 0},
		{AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND, 0, 0, "STAT%s:OPER:SHUT:PROT:COND", 0},
		{AMETEKDC_VAL_STAT_OPER_SHUT_PROT_PTR, 0, 0, "STAT%s:OPER:SHUT:PROT:PTR", 0},
		{AMETEKDC_VAL_STAT_OPER_SHUT_PROT_NTR, 0, 0, "STAT%s:OPER:SHUT:PROT:NTR", 0},
		{AMETEKDC_VAL_STAT_STANDARD_EVENT, 0, 0, "STAT%s:STAN", 0},
		{AMETEKDC_VAL_STAT_STANDARD_ENABLE, 0, 0, "STAT%s:STAN:ENAB", 0},
		{AMETEKDC_VAL_STAT_STATUS_BYTE, 0, 0, "STAT%s:SBYT", 0},
		{AMETEKDC_VAL_STAT_SERVICE_REQUEST, 0, 0, "STAT%s:SREQ", 0},
		{AMETEKDC_VAL_SENS_PROT_INTERLOCK, 0, 0, "SENS%s:PROT:INT", 0},
		{AMETEKDC_VAL_SOUR_VOLT, 0, 0, "SOUR%s:VOLT", AMETEKDC_ATTR_VOLTAGE_LEVEL},
		{AMETEKDC_VAL_SOUR_VOLT_PROT_OVP, 0, 0, "SOUR%s:VOLT:PROT", 0},
		{AMETEKDC_VAL_SOUR_VOLT_PROT_UVP, 0, 0, "SOUR%s:VOLT:PROT:UND", 0},
		{AMETEKDC_VAL_SOUR_CURR, 0, 0, "SOUR%s:CURR", AMETEKDC_ATTR_CURRENT_LIMIT},
		{AMETEKDC_VAL_SOUR_CURR_PROT, 0, 0, "SOUR%s:CURR:PROT", 0},
		{AMETEKDC_VAL_SOUR_COMB_CSH, 0, 0, "SOUR%s:COMB:CSH", 0},
		{AMETEKDC_VAL_PROG_STATE, 0, 0, "PROG%s:STAT", 0},
		{AMETEKDC_VAL_PROG_START_RECORDING, 0, 0, "PROG%s:STAR", 0},
		{AMETEKDC_VAL_PROG_STOP_RECORDING, 0, 0, "PROG%s:STOP", 0},
		{AMETEKDC_VAL_PROG_DELETE_ALL, 0, 0, "PROG%s:DEL", 0},
		{AMETEKDC_VAL_PROG_REPEAT, 0, 0, "PROG%s:REP", 0},
		{AMETEKDC_VAL_PROG_DWELL, 0, 0, "PROG%s:DWEL", 0},
		{AMETEKDC_VAL_PROG_READBACK, 0, 0, "PROG%s:READ", 0},
		{AMETEKDC_VAL_OUTP_PROT_CLEAR, 0, 0, "OUTP%s:PROT:CLE", 0},
		{AMETEKDC_VAL_OUTP_PROT_FOLD, 0, 0, "OUTP%s:PROT:FOLD", 0},
		{AMETEKDC_VAL_OUTP_PROT_FOLD_DELAY, 0, 0, "OUTP%s:PROT:FOLD:DEL", 0},
		{AMETEKDC_VAL_OUTP_SD_POL, 0, 0, "OUTP%s:POL", 0},
		{AMETEKDC_VAL_OUTP_STATE, 0, 0, "OUTP%s:STAT", 0},
		{AMETEKDC_VAL_OUTP_PON, 0, 0, "OUTP%s:PON", 0},
		{AMETEKDC_VAL_OUTP_AUX, 0, 0, "OUTP%s:AUX", 0},
		{AMETEKDC_VAL_OUTP_AUX_PON, 0, 0, "OUTP%s:AUX:PON", 0},
		{AMETEKDC_VAL_MEAS_VOLT, 0, 0, "MEAS%s:VOLT", 0},
		{AMETEKDC_VAL_MEAS_CURR, 0, 0, "MEAS%s:CURR", 0},
		{AMETEKDC_VAL_MEAS_VAP, 0, 0, "MEAS%s:APR", 0},
		{AMETEKDC_VAL_MEAS_VAP_ISOL, 0, 0, "MEAS%s:APR:ISOL", 0},
		{AMETEKDC_VAL_MEAS_CAP, 0, 0, "MEAS%s:APR:CURR", 0},
		{AMETEKDC_VAL_MEAS_CAP_ISOL, 0, 0, "MEAS%s:APR:CURR:ISOL", 0},
		{AMETEKDC_VAL_INITIATE_IMM, 0, 0, "INIT%s:IMM", 0},
		{AMETEKDC_VAL_CAL_RESTORE, 0, 0, "CAL%s:REST", 0},
		{AMETEKDC_VAL_CAL_OVP, 0, 0, "CAL%s:PROT", 0},
		{AMETEKDC_VAL_CAL_PARAM_INPUT, 0, 0, "CAL%s:PAR:INP", 0},
		{AMETEKDC_VAL_CAL_PARAM, 0, 0, "CAL%s:PAR", 0},
		{AMETEKDC_VAL_CAL_PARAM_APR, 0, 0, "CAL%s:PAR:APR", 0},
		{AMETEKDC_VAL_CAL_OUTP_VOLT, 0, 0, "CAL%s:OUTP", 0},
		{AMETEKDC_VAL_CAL_OUTP_CURR, 0, 0, "CAL%s:OUTP:CURR", 0},
		{AMETEKDC_VAL_CAL_OUTP_VAP_VOLT, 0, 0, "CAL%s:OUTP:ANAL", 0},
		{AMETEKDC_VAL_CAL_OUTP_VAP_VOLT_ISOL, 0, 0, "CAL%s:OUTP:ANAL:ISOL", 0},
		{AMETEKDC_VAL_CAL_OUTP_CAP_VOLT, 0, 0, "CAL%s:OUTP:ANAL:CURR", 0},
		{AMETEKDC_VAL_CAL_OUTP_CAP_VOLT_ISOL, 0, 0, "CAL%s:OUTP:ANAL:CURR:ISOL", 0},
		{AMETEKDC_VAL_CAL_OUTP_VAP_RES, 0, 0, "CAL%s:OUTP:ANAL:RES", 0},
		{AMETEKDC_VAL_CAL_OUTP_VAP_RES_ISOL, 0, 0, "CAL%s:OUTP:ANAL:RES:ISOL", 0},
		{AMETEKDC_VAL_CAL_OUTP_CAP_RES, 0, 0, "CAL%s:OUTP:ANAL:CURR:RES", 0},
		{AMETEKDC_VAL_CAL_OUTP_CAP_RES_ISOL, 0, 0, "CAL%s:OUTP:ANAL:CURR:RES:ISOL", 0},
		{AMETEKDC_VAL_CAL_MONITOR_VOLT, 0, 0, "CAL%s:MON", 0},
		{AMETEKDC_VAL_CAL_MONITOR_VOLT_ISOL, 0, 0, "CAL%s:MON:ISOL", 0},
		{AMETEKDC_VAL_CAL_MONITOR_CURR, 0, 0, "CAL%s:MON:CURR", 0},
		{AMETEKDC_VAL_CAL_MONITOR_CURR_ISOL, 0, 0, "CAL%s:MON:CURR:ISOL", 0},
		{AMETEKDC_VAL_CAL_INPUT_VAP, 0, 0, "CAL%s:INP:ANAL", 0},
		{AMETEKDC_VAL_CAL_INPUT_CAP, 0, 0, "CAL%s:INP:ANAL:CURR", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable scpiCmdStringRangeTable =
	{
		IVI_VAL_DISCRETE,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        scpiCmdStringRangeTableEntries,
	};

static IviRangeTableEntry attrFpInactivityTimeoutRangeTableEntries[] =
	{
		{1000, 20000, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable attrFpInactivityTimeoutRangeTable =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        attrFpInactivityTimeoutRangeTableEntries,
	};

#ifdef __AMETEKDC_SPECIFIC_ADDRESS_HANDLING__
#define SETUP_MCHADDR_TAG      "MCHADDR"
#endif


//********************* Output Current Limit Range Table *****************************
static IviRangeTableEntry outputCurrLimitRangeTable1_4Entries[] =
	{
		{0, (1.4 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable1_4 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable1_4Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable2_8Entries[] =
	{
		{0, (2.8 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable2_8 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable2_8Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable5_6Entries[] =
	{
		{0, (5.6 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable5_6 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable5_6Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable8_5Entries[] =
	{
		{0, (8.5 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable8_5 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable8_5Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable10_5Entries[] =
	{
		{0, (10.5 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable10_5 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable10_5Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable14Entries[] =
	{
		{0, (14 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable14 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable14Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable21Entries[] =
	{
		{0, (21 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable21 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable21Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable25Entries[] =
	{
		{0, (25 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable25 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable25Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable42Entries[] =
	{
		{0, (42 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable42 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable42Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable70Entries[] =
	{
		{0, (70 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable70 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable70Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable100Entries[] =
	{
		{0, (100  * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable100 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable100Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable110Entries[] =
	{
		{0, (110 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable110 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable110Entries,
	};
//********************* Output Current Limit Range Table *****************************  
	  			 
//********************************** Added for XG1700 ********************************
//********************* Output Current Limit Range Table *****************************  
static IviRangeTableEntry outputCurrLimitRangeTable11_2Entries[] =
	{
		{0, (11.2 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable11_2 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable11_2Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable17Entries[] =
	{
		{0, (17 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable17 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable17Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable28Entries[] =
	{
		{0, (28 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable28 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable28Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable50Entries[] =
	{
		{0, (50 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable50 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable50Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable82Entries[] =
	{
		{0, (82 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable82 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable82Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable140Entries[] =
	{
		{0, (140 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable140 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable140Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable200Entries[] =
	{
		{0, (200 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable200 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable200Entries,
	};

static IviRangeTableEntry outputCurrLimitRangeTable220Entries[] =
	{
		{0, (220 * MAX_OUTPUT_CURRENT), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputCurrLimitRangeTable220 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputCurrLimitRangeTable220Entries,
	};
//********************* Output Current Limit Range Table *****************************  
//********************************** Added for XG1700 ********************************


/*****************************************************************************
 *--------------------- Output Limit Range Tables ---------------------------*
 *****************************************************************************/
/* The following range tables are used for all voltage, current, and power
   setpoints. They are all named the same, except for the number at the end.
   This number signifies the nominal maximum value for the output parameter. 
   ie. outputLimitRangeTable10 would be used for voltage settings on a 10V 
   unit. The actual maximum value is 103% of the nominal value.             */
   
//********************* Output Voltage Limit Range Table *****************************  
static IviRangeTableEntry outputVoltLimitRangeTable6Entries[] =
	{
		{0, (6  * MAX_OUTPUT_VOLTAGE), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputVoltLimitRangeTable6 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputVoltLimitRangeTable6Entries,
	};

static IviRangeTableEntry outputVoltLimitRangeTable8Entries[] =
	{
		{0, (8 * MAX_OUTPUT_VOLTAGE), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputVoltLimitRangeTable8 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputVoltLimitRangeTable8Entries,
	};

static IviRangeTableEntry outputVoltLimitRangeTable12Entries[] =
	{
		{0, (12  * MAX_OUTPUT_VOLTAGE), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputVoltLimitRangeTable12 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputVoltLimitRangeTable12Entries,
	};
	 
static IviRangeTableEntry outputVoltLimitRangeTable20Entries[] =
	{
		{0, (20 * MAX_OUTPUT_VOLTAGE), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputVoltLimitRangeTable20 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputVoltLimitRangeTable20Entries,
	};

static IviRangeTableEntry outputVoltLimitRangeTable33Entries[] =
	{
		{0, (33  * MAX_OUTPUT_VOLTAGE), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputVoltLimitRangeTable33 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputVoltLimitRangeTable33Entries,
	};

static IviRangeTableEntry outputVoltLimitRangeTable40Entries[] =
	{
		{0, (40  * MAX_OUTPUT_VOLTAGE), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputVoltLimitRangeTable40 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputVoltLimitRangeTable40Entries,
	};

static IviRangeTableEntry outputVoltLimitRangeTable60Entries[] =
	{
		{0, (60  * MAX_OUTPUT_VOLTAGE), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputVoltLimitRangeTable60 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputVoltLimitRangeTable60Entries,
	};
	
static IviRangeTableEntry outputVoltLimitRangeTable80Entries[] =
	{
		{0, (80 * MAX_OUTPUT_VOLTAGE), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputVoltLimitRangeTable80 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputVoltLimitRangeTable80Entries,
	};

static IviRangeTableEntry outputVoltLimitRangeTable100Entries[] =
	{
		{0, (100 * MAX_OUTPUT_VOLTAGE), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputVoltLimitRangeTable100 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputVoltLimitRangeTable100Entries,
	};

static IviRangeTableEntry outputVoltLimitRangeTable150Entries[] =
	{
		{0, (150 * MAX_OUTPUT_VOLTAGE), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputVoltLimitRangeTable150 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputVoltLimitRangeTable150Entries,
	};
 
static IviRangeTableEntry outputVoltLimitRangeTable300Entries[] =
	{
		{0, (300 * MAX_OUTPUT_VOLTAGE), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputVoltLimitRangeTable300 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputVoltLimitRangeTable300Entries,
	};
	 
static IviRangeTableEntry outputVoltLimitRangeTable600Entries[] =
	{
		{0, (600  * MAX_OUTPUT_VOLTAGE), 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable outputVoltLimitRangeTable600 =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        outputVoltLimitRangeTable600Entries,
	};
//********************* Output Voltage Limit Range Table *****************************  

/* Family members single-output, single-range, triggered */
typedef enum eSupportedModels
{
   // XTR 850W Models
   XTR6_110,
   XTR8_100,
   XTR12_70,
   XTR20_42,
   XTR33_25,
   XTR40_21,
   XTR60_14,
   XTR80_10_5,
   XTR100_8_5,
   XTR150_5_6,
   XTR300_2_8,
   XTR600_1_4,
   
   // XTR 1700W Models
   XTR6_220,
   XTR8_200,
   XTR12_140,
   XTR20_84,
   XTR33_50,
   XTR40_42,
   XTR60_28,
   XTR80_21,
   XTR100_17,
   XTR150_11_2,
   XTR300_5_6,
   XTR600_2_8,
   
   XTR_NUMBER_MODELS		/* Must appear last */
   
}teSupportedModels;

										   
#define AMETEKDC_MODEL_NOT_FOUND    -1L
#define AMETEKDC_DEFAULT_MODEL_TYPE    XTR100_8_5

static IviStringValueTable AmetekDC_ModelTable =
{
   // XG 850W Models ****************
   {XTR6_110, 		"XG 6-110.0"	},
   {XTR8_100, 		"XG 8-100.0"	},
   {XTR12_70, 		"XG 12-70.0"	},
   {XTR20_42, 		"XG 20-42.0"	},
   {XTR33_25, 		"XG 33-25.0"	},
   {XTR40_21, 		"XG 40-21.0"	},
   {XTR60_14, 		"XG 60-14.0"	},
   {XTR80_10_5, 	"XG 80-10.5"	},
   {XTR100_8_5, 	"XG 100-8.5"	},
   {XTR150_5_6, 	"XG 150-5.6"	},
   {XTR300_2_8, 	"XG 300-2.8"	},
   {XTR600_1_4, 	"XG 600-1.4"	},
   
   {XTR6_110, 		"XG 6-110"		},
   {XTR8_100, 		"XG 8-100"		},
   {XTR12_70, 		"XG 12-70"		},
   {XTR20_42, 		"XG 20-42"		},
   {XTR33_25, 		"XG 33-25"		},
   {XTR40_21, 		"XG 40-21"		},
   {XTR60_14, 		"XG 60-14"		},
   
   // XTR 850W Models ***************
   {XTR6_110, 		"XTR 6-110.0"	},
   {XTR8_100, 		"XTR 8-100.0"	},
   {XTR12_70, 		"XTR 12-70.0"	},
   {XTR20_42, 		"XTR 20-42.0"	},
   {XTR33_25, 		"XTR 33-25.0"	},
   {XTR40_21,	 	"XTR 40-21.0"	},
   {XTR60_14, 		"XTR 60-14.0"	},
   {XTR80_10_5, 	"XTR 80-10.5"	},
   {XTR100_8_5, 	"XTR 100-8.5"	},
   {XTR150_5_6, 	"XTR 150-5.6"	},
   {XTR300_2_8, 	"XTR 300-2.8"	},
   {XTR600_1_4, 	"XTR 600-1.4"	},
   
   {XTR6_110, 		"XTR 6-110"		},
   {XTR8_100, 		"XTR 8-100"		},
   {XTR12_70, 		"XTR 12-70"		},
   {XTR20_42, 		"XTR 20-42"		},
   {XTR33_25, 		"XTR 33-25"		},
   {XTR40_21,	 	"XTR 40-21"		},
   {XTR60_14, 		"XTR 60-14"		},
   
    
   // XG 1700W Models ***************
   {XTR6_220, 		"XG 6-220.0"	},
   {XTR8_200, 		"XG 8-200.0"	},
   {XTR12_140, 		"XG 12-140.0"	},
   {XTR20_84, 		"XG 20-84.0"	},
   {XTR33_50, 		"XG 33-50.0"	},
   {XTR40_42, 		"XG 40-42.0"	},
   {XTR60_28, 		"XG 60-28.0"	},
   {XTR80_21, 		"XG 80-21.0"	},
   {XTR100_17, 		"XG 100-17.0"	},
   {XTR150_11_2,	"XG 150-11.2"	},
   {XTR300_5_6, 	"XG 300-5.6"	},
   {XTR600_2_8, 	"XG 600-2.8"	},
   
   {XTR6_220, 		"XG 6-220"		},
   {XTR8_200, 		"XG 8-200"		},
   {XTR12_140,		"XG 12-140"		},
   {XTR20_84, 		"XG 20-84"		},
   {XTR33_50, 		"XG 33-50"		},
   {XTR40_42, 		"XG 40-42"		},
   {XTR60_28, 		"XG 60-28"		},
   {XTR80_21, 		"XG 80-21"		},
   {XTR100_17,		"XG 100-17"		},
   
   
   // XTR 1700W Models **************
   {XTR6_220, 		"XTR 6-220.0"	},
   {XTR8_200, 		"XTR 8-200.0"	},
   {XTR12_140, 		"XTR 12-140.0"	},
   {XTR20_84, 		"XTR 20-84.0"	},
   {XTR33_50, 		"XTR 33-50.0"	},
   {XTR40_42, 		"XTR 40-42.0"	},
   {XTR60_28, 		"XTR 60-28.0"	},
   {XTR80_21, 		"XTR 80-21.0"	},
   {XTR100_17, 		"XTR 100-17.0"	},
   {XTR150_11_2,	"XTR 150-11.2"	},
   {XTR300_5_6, 	"XTR 300-5.6"	},
   {XTR600_2_8, 	"XTR 600-2.8"	},
   
   {XTR6_220, 		"XTR 6-220"		},
   {XTR8_200, 		"XTR 8-200"		},
   {XTR12_140,		"XTR 12-140"	},
   {XTR20_84, 		"XTR 20-84"		},
   {XTR33_50, 		"XTR 33-50"		},
   {XTR40_42, 		"XTR 40-42"		},
   {XTR60_28, 		"XTR 60-28"		},
   {XTR80_21, 		"XTR 80-21"		},
   {XTR100_17,		"XTR 100-17"	},
   
   {VI_NULL,   VI_NULL}
};

// Structure for storing the range tables for the parameters of the power supply.
typedef struct zModelOutput{
	ViAddr addrVoltRangeTbl;
	ViAddr addrCurrRangeTbl;
	ViAddr addrOvpRangeTbl;
}tzModelOutput;

// Look up table used to assign the correct range tables for each model.
// Note: that the order of this array must match the enum teSupportedModels
// as these values will be used to index this table.
static const tzModelOutput AmetekDC_kazModelOutputRangeTable[] =
{
   // XG 850 Models
   {&outputVoltLimitRangeTable6, &outputCurrLimitRangeTable110, &outputProtOvpRangTable6},      /* XG6_110 */
   {&outputVoltLimitRangeTable8, &outputCurrLimitRangeTable100, &outputProtOvpRangTable8},      /* XG8_100 */
   {&outputVoltLimitRangeTable12, &outputCurrLimitRangeTable70, &outputProtOvpRangTable12},      /* XG12_70 */
   {&outputVoltLimitRangeTable20, &outputCurrLimitRangeTable42, &outputProtOvpRangTable20},      /* XG20_42 */
   {&outputVoltLimitRangeTable33, &outputCurrLimitRangeTable25, &outputProtOvpRangTable33},      /* XG33_25 */
   {&outputVoltLimitRangeTable40, &outputCurrLimitRangeTable21, &outputProtOvpRangTable40},      /* XG40_21 */
   {&outputVoltLimitRangeTable60, &outputCurrLimitRangeTable14, &outputProtOvpRangTable60},      /* XG60_14 */
   {&outputVoltLimitRangeTable80, &outputCurrLimitRangeTable10_5, &outputProtOvpRangTable80},      /* XG80_10_5 */
   {&outputVoltLimitRangeTable100, &outputCurrLimitRangeTable8_5, &outputProtOvpRangTable100},      /* XG100_8_5 */
   {&outputVoltLimitRangeTable150, &outputCurrLimitRangeTable5_6, &outputProtOvpRangTable150},      /* XG150_5_6 */
   {&outputVoltLimitRangeTable300, &outputCurrLimitRangeTable2_8, &outputProtOvpRangTable300},      /* XG300_2_8 */
   {&outputVoltLimitRangeTable600, &outputCurrLimitRangeTable1_4, &outputProtOvpRangTable600},      /* XG600_1_4 */
   
   // XG 1700 Models 
   {&outputVoltLimitRangeTable6, &outputCurrLimitRangeTable220, &outputProtOvpRangTable6},      /* XG6_220 */
   {&outputVoltLimitRangeTable8, &outputCurrLimitRangeTable200, &outputProtOvpRangTable8},      /* XG8_200 */
   {&outputVoltLimitRangeTable12, &outputCurrLimitRangeTable140, &outputProtOvpRangTable12},      /* XG12_140 */
   {&outputVoltLimitRangeTable20, &outputCurrLimitRangeTable82, &outputProtOvpRangTable20},      /* XG20_82 */
   {&outputVoltLimitRangeTable33, &outputCurrLimitRangeTable50, &outputProtOvpRangTable33},      /* XG33_50 */
   {&outputVoltLimitRangeTable40, &outputCurrLimitRangeTable42, &outputProtOvpRangTable40},      /* XG40_42 */
   {&outputVoltLimitRangeTable60, &outputCurrLimitRangeTable28, &outputProtOvpRangTable60},      /* XG60_28 */
   {&outputVoltLimitRangeTable80, &outputCurrLimitRangeTable21, &outputProtOvpRangTable80},      /* XG80_21 */
   {&outputVoltLimitRangeTable100, &outputCurrLimitRangeTable17, &outputProtOvpRangTable100},      /* XG100_17 */
   {&outputVoltLimitRangeTable150, &outputCurrLimitRangeTable11_2, &outputProtOvpRangTable150},      /* XG150_11_2 */
   {&outputVoltLimitRangeTable300, &outputCurrLimitRangeTable5_6, &outputProtOvpRangTable300},      /* XG300_5_6 */
   {&outputVoltLimitRangeTable600, &outputCurrLimitRangeTable2_8, &outputProtOvpRangTable600}      /* XG600_2_8 */
};

/*****************************************************************************
 *-------------- Utility Function Declarations (Non-Exported) ---------------*
 *****************************************************************************/
static ViStatus AmetekDC_InitAttributes ( ViSession vi, ViInt32 modelType, ViBoolean IDQuery );

static ViStatus AmetekDC_DefaultInstrSetup( ViSession openInstrSession );

static ViStatus AmetekDC_CheckStatus( ViSession vi, ViConstString channelName );

static ViStatus AmetekDC_Delay( ViSession vi ); 

static ViStatus AmetekDC_WaitForOPC( ViSession vi, ViInt32 maxTime );
                                
static ViStatus AmetekDC_GetDriverSetupOption( ViSession vi, ViInt32 *modelType,
                                             ViChar channelString[] );                                 

static ViStatus AmetekDC_GetModelFromBuffer( ViSession vi, ViConstString buffer,
                                           ViInt32 *modelType );
                                           
static ViStatus AmetekDC_SetRangeTableAddress( ViSession vi, 
                                             ViConstString channelString );
                                             
static ViStatus AmetekDC_VerifyChannelName( ViSession vi, 
                                          ViConstString channelName,
                                          ViBoolean *isValid );                                             
                                           
static void RemoveSurroundingWhiteSpace( ViChar string[] ); 


/*****************************************************************************
 *---------- Configure Protection Declarations (Non-Exported) ---------------*
 *****************************************************************************/
 
static ViStatus AmetekDC_ConfigureOCP( ViSession vi, ViConstString channelName,
                                     ViBoolean behavior, ViReal64 limit );
                                     
static ViStatus AmetekDC_ConfigureUVP( ViSession vi, ViConstString channelName,
                                     ViBoolean behavior, ViReal64 limit );
                                     
                                                                       

/*****************************************************************************
 *----------------- Callback Declarations (Non-Exported) --------------------*
 *****************************************************************************/

/*- Global Session Callbacks --------------------------------------------*/
static ViStatus _VI_FUNC AmetekDC_CheckStatusCallback( 
                                             ViSession vi, ViSession io,
                                             ViConstString channelName );

/*- Attribute callbacks -------------------------------------------------*/
static ViStatus _VI_FUNC AmetekDCAttrDriverRevision_ReadCallback( 
                                             ViSession vi,
                                             ViSession io, 
                                             ViConstString channelName,
                                             ViAttr attributeId, 
                                             const ViConstString cacheValue );
                                                                  
static ViStatus _VI_FUNC AmetekDCAttrInstrumentManufacturer_ReadCallback(
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             const ViConstString cacheValue );
                                                                          
static ViStatus _VI_FUNC AmetekDCAttrInstrumentModel_ReadCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             const ViConstString cacheValue );

static ViStatus _VI_FUNC AmetekDCAttrInstrumentFirmwareRevision_ReadCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             const ViConstString cacheValue );
                                                                    
static ViStatus _VI_FUNC AmetekDCAttrOutputEnabled_WriteCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViBoolean value );
                                                                 
static ViStatus _VI_FUNC AmetekDCAttrOutputEnabled_ReadCallback(
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViBoolean *value );
                                                                 
static ViStatus _VI_FUNC AmetekDCAttrIdQueryResponse_ReadCallback( 
                                             ViSession vi,
                                             ViSession io, 
                                             ViConstString channelName, 
                                             ViAttr attributeId, 
                                             const ViConstString cacheValue );

static ViStatus _VI_FUNC AmetekDCAttrFoldDelay_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 *value );

static ViStatus _VI_FUNC AmetekDCAttrFoldDelay_WriteCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 value );
                                                              
static ViStatus _VI_FUNC AmetekDCAttrVoltageLevel_WriteCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViReal64 value );
                                                                
static ViStatus _VI_FUNC AmetekDCAttrVoltageLevel_ReadCallback(
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViReal64* value );

static ViStatus _VI_FUNC AmetekDCAttrOVPLimit_WriteCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 value );
                                                            
static ViStatus _VI_FUNC AmetekDCAttrOVPLimit_ReadCallback(
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViReal64* value );
                                                
static ViStatus _VI_FUNC AmetekDCAttrCurrentLimit_WriteCallback(
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViReal64 value );
                                                                
static ViStatus _VI_FUNC AmetekDCAttrCurrentLimit_ReadCallback(
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViReal64* value );
 
static ViStatus _VI_FUNC AmetekDCAttrFoldbackSetting_WriteCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViConstString value );
                                                                
static ViStatus _VI_FUNC AmetekDCAttrFoldbackSetting_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             const ViConstString cacheValue );

static ViStatus _VI_FUNC AmetekDCAttrOcpLimit_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 *value );

static ViStatus _VI_FUNC AmetekDCAttrOcpLimit_WriteCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 value );
                                                             

static ViStatus _VI_FUNC AmetekDCAttrUvpLimit_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 *value );

static ViStatus _VI_FUNC AmetekDCAttrUvpLimit_WriteCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 value );



static ViStatus _VI_FUNC AmetekDCAttrCurrentLimitBehavior_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViInt32 *value );

static ViStatus _VI_FUNC AmetekDCAttrCurrentLimitBehavior_WriteCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViInt32 value );
													
static ViStatus _VI_FUNC AmetekDCAttrUvpBehavior_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean *value );

static ViStatus _VI_FUNC AmetekDCAttrUvpBehavior_WriteCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean value );

static ViStatus _VI_FUNC AmetekDCAttrCurrentShareMode_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             const ViConstString cacheValue );

static ViStatus _VI_FUNC AmetekDCAttrCurrentShareMode_WriteCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViConstString value );

static ViStatus _VI_FUNC AmetekDCAttrAcFailResponse_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean *value );

static ViStatus _VI_FUNC AmetekDCAttrAcFailResponse_WriteCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean value );

static ViStatus _VI_FUNC AmetekDCAttrOverTemperatureResponse_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean *value );
                                             
static ViStatus _VI_FUNC AmetekDCAttrOverTemperatureResponse_WriteCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean value );

static ViStatus _VI_FUNC AmetekDCAttrOutputEnabledPowerOn_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean *value );

static ViStatus _VI_FUNC AmetekDCAttrOutputEnabledPowerOn_WriteCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean value );



static ViStatus _VI_FUNC AmetekDCAttrVoltageAnalogProgrammingRange_ReadCallback (ViSession vi,
                                                                               ViSession io,
                                                                               ViConstString channelName,
                                                                               ViAttr attributeId,
                                                                               ViReal64 *value);

static ViStatus _VI_FUNC AmetekDCAttrVoltAnalogProgrammingRange_WriteCallback (ViSession vi,
                                                                             ViSession io,
                                                                             ViConstString channelName,
                                                                             ViAttr attributeId,
                                                                             ViReal64 value);





static ViStatus _VI_FUNC AmetekDCAttrRemoteMode_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             const ViConstString cacheValue );

static ViStatus _VI_FUNC AmetekDCAttrRemoteMode_WriteCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViConstString value );

static ViStatus _VI_FUNC AmetekDCAttrRemoteSourceVolt_ReadCallback (ViSession vi,
                                                                  ViSession io,
                                                                  ViConstString channelName,
                                                                  ViAttr attributeId,
                                                                  const ViConstString cacheValue);

static ViStatus _VI_FUNC AmetekDCAttrRemoteSourceVolt_WriteCallback (ViSession vi,
                                                                   ViSession io,
                                                                   ViConstString channelName,
                                                                   ViAttr attributeId,
                                                                   ViConstString value);

static ViStatus _VI_FUNC AmetekDCAttrRemoteSourceCurr_ReadCallback (ViSession vi,
                                                                  ViSession io,
                                                                  ViConstString channelName,
                                                                  ViAttr attributeId,
                                                                  const ViConstString cacheValue);

static ViStatus _VI_FUNC AmetekDCAttrRemoteSourceCurr_WriteCallback (ViSession vi,
                                                                   ViSession io,
                                                                   ViConstString channelName,
                                                                   ViAttr attributeId,
                                                                   ViConstString value);
   

static ViStatus _VI_FUNC AmetekDCAttrModelType_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViInt32 *value );

static ViStatus _VI_FUNC AmetekDCAttrVoltageLevel_RangeTableCallback(
                                             ViSession vi,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             IviRangeTablePtr *rangeTablePtr );

static ViStatus _VI_FUNC AmetekDCAttrCurrentLimit_RangeTableCallback(
                                             ViSession vi,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             IviRangeTablePtr *rangeTablePtr );


static ViStatus _VI_FUNC AmetekDCAttrMultichannelFirmwareRevision_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             const ViConstString cacheValue );

static ViStatus _VI_FUNC AmetekDCAttrMultichannelModel_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             const ViConstString cacheValue );



static ViStatus _VI_FUNC AmetekDCAttrFoldbackResponse_ReadCallback (ViSession vi,
                                                                  ViSession io,
                                                                  ViConstString channelName,
                                                                  ViAttr attributeId,
                                                                  ViBoolean *value);

static ViStatus _VI_FUNC AmetekDCAttrFoldbackResponse_WriteCallback (ViSession vi,
                                                                   ViSession io,
                                                                   ViConstString channelName,
                                                                   ViAttr attributeId,
                                                                   ViBoolean value);

static ViStatus _VI_FUNC AmetekDCAttrOverTemperatureSlaveResponse_ReadCallback (ViSession vi,
                                                                              ViSession io,
                                                                              ViConstString channelName,
                                                                              ViAttr attributeId,
                                                                              ViBoolean *value);

static ViStatus _VI_FUNC AmetekDCAttrOverTemperatureSlaveResponse_WriteCallback (ViSession vi,
                                                                               ViSession io,
                                                                               ViConstString channelName,
                                                                               ViAttr attributeId,
                                                                               ViBoolean value);

static ViStatus _VI_FUNC AmetekDCAttrCurrAnalogProgrammingRange_ReadCallback (ViSession vi,
                                                                            ViSession io,
                                                                            ViConstString channelName,
                                                                            ViAttr attributeId,
                                                                            ViReal64 *value);

static ViStatus _VI_FUNC AmetekDCAttrCurrAnalogProgrammingRange_WriteCallback (ViSession vi,
                                                                             ViSession io,
                                                                             ViConstString channelName,
                                                                             ViAttr attributeId,
                                                                             ViReal64 value);

static ViStatus _VI_FUNC AmetekDCAttrFpInactivityTimeout_ReadCallback (ViSession vi,
                                                                     ViSession io,
                                                                     ViConstString channelName,
                                                                     ViAttr attributeId,
                                                                     ViInt32 *value);

static ViStatus _VI_FUNC AmetekDCAttrFpInactivityTimeout_WriteCallback (ViSession vi,
                                                                      ViSession io,
                                                                      ViConstString channelName,
                                                                      ViAttr attributeId,
                                                                      ViInt32 value);

static ViStatus _VI_FUNC AmetekDCAttrOVPEnabled_ReadCallback (ViSession vi,
                                                            ViSession io,
                                                            ViConstString channelName,
                                                            ViAttr attributeId,
                                                            ViBoolean *value);

static ViStatus _VI_FUNC AmetekDCAttrOVPEnabled_WriteCallback (ViSession vi,
                                                             ViSession io,
                                                             ViConstString channelName,
                                                             ViAttr attributeId,
                                                             ViBoolean value);

static ViStatus _VI_FUNC AmetekDCAttrAuxEnabledPowerOn_ReadCallback (ViSession vi,
                                                                   ViSession io,
                                                                   ViConstString channelName,
                                                                   ViAttr attributeId,
                                                                   ViBoolean *value);

static ViStatus _VI_FUNC AmetekDCAttrAuxEnabledPowerOn_WriteCallback (ViSession vi,
                                                                    ViSession io,
                                                                    ViConstString channelName,
                                                                    ViAttr attributeId,
                                                                    ViBoolean value);

static ViStatus _VI_FUNC AmetekDCAttrAuxEnabled_ReadCallback (ViSession vi,
                                                            ViSession io,
                                                            ViConstString channelName,
                                                            ViAttr attributeId,
                                                            ViBoolean *value);

static ViStatus _VI_FUNC AmetekDCAttrAuxEnabled_WriteCallback (ViSession vi,
                                                             ViSession io,
                                                             ViConstString channelName,
                                                             ViAttr attributeId,
                                                             ViBoolean value);

static ViStatus _VI_FUNC AmetekDCAttrExtShutdownLogic_ReadCallback (ViSession vi,
                                                                  ViSession io,
                                                                  ViConstString channelName,
                                                                  ViAttr attributeId,
                                                                  ViBoolean *value);

static ViStatus _VI_FUNC AmetekDCAttrExtShutdownLogic_WriteCallback (ViSession vi,
                                                                   ViSession io,
                                                                   ViConstString channelName,
                                                                   ViAttr attributeId,
                                                                   ViBoolean value);


static ViStatus _VI_FUNC AmetekDCAttrAutoSequenceDwellTime_ReadCallback (ViSession vi,
                                                                       ViSession io,
                                                                       ViConstString channelName,
                                                                       ViAttr attributeId,
                                                                       ViInt32 *value);

static ViStatus _VI_FUNC AmetekDCAttrAutoSequenceDwellTime_WriteCallback (ViSession vi,
                                                                        ViSession io,
                                                                        ViConstString channelName,
                                                                        ViAttr attributeId,
                                                                        ViInt32 value);

static ViStatus _VI_FUNC AmetekDCAttrProtectionMask_ReadCallback (ViSession vi,
                                                                ViSession io,
                                                                ViConstString channelName,
                                                                ViAttr attributeId,
                                                                ViInt32 *value);

static ViStatus _VI_FUNC AmetekDCAttrProtectionMask_WriteCallback (ViSession vi,
                                                                 ViSession io,
                                                                 ViConstString channelName,
                                                                 ViAttr attributeId,
                                                                 ViInt32 value);

static ViStatus _VI_FUNC AmetekDCAttrConfigurableDelay_CheckCallback (ViSession vi,
                                                                    ViConstString channelName,
                                                                    ViAttr attributeId,
                                                                    ViReal64 value);

static ViStatus _VI_FUNC AmetekDCAttrOvpLimit_RangeTableCallback (ViSession vi,
                                                                ViConstString channelName,
                                                                ViAttr attributeId,
                                                                IviRangeTablePtr *rangeTablePtr);



static ViStatus _VI_FUNC AmetekDCGetProtectionLatch_CallbackHelper(
                                             ViSession vi,
                                             ViSession io,								                 
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean *value,
                                             ViUInt16 flagPosition);


static ViStatus _VI_FUNC AmetekDCSetProtectionLatch_CallbackHelper(
                                             ViSession vi,
                                             ViSession io,								                 
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean value,
                                             ViUInt16 flagPosition);


static ViStatus AmetekDCQueryID_CallbackHelper( 
                                             ViSession vi, 
                                             ViSession io, 
                                             ViConstString channelName,
                                             ViInt32 bufferSize,
                                             ViChar buffer[] );
                                             
                                             
/*------------ General Communication Functions ------------------------------*/

ViStatus _VI_FUNC AmetekDC_SendCmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
                                             ViUInt32 cmdIdx,
                                             ViConstString cmdParameters );
                                             
ViStatus _VI_FUNC AmetekDC_SendBooleanCmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,  
                                             ViBoolean value);

ViStatus _VI_FUNC AmetekDC_SendInt16Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,  
                                             ViInt16 value);

ViStatus _VI_FUNC AmetekDC_SendUInt16Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,  
                                             ViUInt16 value);

ViStatus _VI_FUNC AmetekDC_SendInt32Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,  
                                             ViInt32 value);

ViStatus _VI_FUNC AmetekDC_SendUInt32Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,  
                                             ViUInt32 value);

ViStatus _VI_FUNC AmetekDC_SendReal64Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,  
                                             ViReal64 value);

ViStatus _VI_FUNC AmetekDC_SendStringCmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
							ViUInt32 cmdIdx,                                             
                                             ViConstString cmdParameters);                                             
                                             
/*****************************************************************************
 *------------ User-Callable Functions(Exportable Functions ) ---------------*
 *****************************************************************************/

/*****************************************************************************
 * Function: AmetekDC_init   
 * Purpose:  VXIplug&play required function.  Calls the   
 *           AmetekDC_InitWithOptions function. 
 *
 * NOTE: Required function for IVI Instrument Driver.
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_init( ViRsrc resourceName, ViBoolean IDQuery,
                               ViBoolean resetDevice, ViSession *newVi )
{   
   ViStatus    error = VI_SUCCESS;

   if( newVi == VI_NULL )
   {
      Ivi_SetErrorInfo( VI_NULL, VI_FALSE, IVI_ERROR_INVALID_PARAMETER, 
                        VI_ERROR_PARAMETER4, 
                        "Null address for Instrument Handle" );
                        
      checkErr( IVI_ERROR_INVALID_PARAMETER );
   }

   checkErr( AmetekDC_InitWithOptions( resourceName, IDQuery, resetDevice, 
                                     "", newVi ) );

Error:  
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_InitWithOptions                                       
 * Purpose:  This function creates a new IVI session and calls the 
 *           IviInit function.  
 *
 * NOTE: Required function for IVI Instrument Driver.
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_InitWithOptions( ViRsrc resourceName, 
                                          ViBoolean IDQuery,
                                          ViBoolean resetDevice, 
                                          ViChar optionString[], 
                                          ViSession *newVi )
{   
   ViStatus    error = VI_SUCCESS;
   ViSession   vi = VI_NULL;
   ViChar      newResourceName[ IVI_MAX_MESSAGE_BUF_SIZE ];
   ViChar      newOptionString[ IVI_MAX_MESSAGE_BUF_SIZE ];
   ViBoolean   isLogicalName;

   if( newVi == VI_NULL )
   {
      Ivi_SetErrorInfo( VI_NULL, VI_FALSE, IVI_ERROR_INVALID_PARAMETER, 
                        VI_ERROR_PARAMETER5, 
                        "Null address for Instrument Handle" );
                        
      checkErr( IVI_ERROR_INVALID_PARAMETER );
   }

   *newVi = VI_NULL;
    
   checkErr( Ivi_GetInfoFromResourceName( resourceName, 
                                         ( ViString )optionString, 
                                          newResourceName,
                                          newOptionString, &isLogicalName ) );
    
   /* create a new interchangeable driver */
   checkErr( Ivi_SpecificDriverNew( "AmetekDC", newOptionString, &vi ) );
   
   /* init the driver */
   checkErr( AmetekDC_IviInit( newResourceName, IDQuery, resetDevice, vi ) ); 
   
   if( isLogicalName )
   {   
      checkErr( Ivi_ApplyDefaultSetup( vi ) );
   }
   
   *newVi = vi;
    
Error:
   if( error < VI_SUCCESS ) 
   {
       Ivi_Dispose( vi );
   }
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_IviInit                                                       
 * Purpose:  This function is called by AmetekDC_InitWithOptions  
 *           or by an IVI class driver.  This function initializes the I/O 
 *           interface, optionally resets the device, optionally performs an
 *           ID query, and sends a default setup to the instrument.         
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_IviInit( ViRsrc resourceName, ViBoolean IDQuery,
                                  ViBoolean reset, ViSession vi )
{
   ViStatus    error = VI_SUCCESS;
   ViSession   io = VI_NULL;
    
   ViChar      rdBuffer[ BUFFER_SIZE ] = "";
   ViChar      channelString[ BUFFER_SIZE ];
   ViInt32     modelType = AMETEKDC_MODEL_NOT_FOUND;
   ViUInt16    interfaceType = VI_INTF_GPIB;
   ViUInt16    busAddress = 2;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   checkErr( AmetekDC_GetDriverSetupOption( vi, &modelType, channelString ) );
  
   if( !Ivi_Simulating( vi ) )
   {
      ViSession   rmSession = VI_NULL;
        
      /* Open instrument session */
      checkErr( Ivi_GetAttributeViSession( vi, VI_NULL, 
                                           IVI_ATTR_VISA_RM_SESSION, 0,
                                           &rmSession ) );
                                           
      viCheckErr( viOpen( rmSession, resourceName, VI_NULL, VI_NULL, &io ) );
      /* io session owned by driver now */

      // Assign the VISA IO session to the IVI session being used.
      checkErr( Ivi_SetAttributeViSession( vi, VI_NULL, IVI_ATTR_IO_SESSION, 
                                           0, io ) );  

      /* Configure VISA Formatted I/O */
      viCheckErr( viSetAttribute( io, VI_ATTR_TMO_VALUE, 1000 ) );
      
      viCheckErr( viSetBuf( io, VI_READ_BUF | VI_WRITE_BUF, 4096 ) );
      
      viCheckErr( viSetAttribute( io, VI_ATTR_WR_BUF_OPER_MODE, 
                                  VI_FLUSH_ON_ACCESS ) );
                                  
      viCheckErr( viSetAttribute( io, VI_ATTR_RD_BUF_OPER_MODE, 
                                  VI_FLUSH_ON_ACCESS ) );      
	  
      // Configure VISA IO termination character. 
      viCheckErr( viSetAttribute( io, VI_ATTR_TERMCHAR, AMETEKDC_VAL_RESP_TERM_CHAR ) );

      // Configure VISA IO to enable using the termination character.  
      viCheckErr( viSetAttribute( io, VI_ATTR_TERMCHAR_EN, VI_TRUE ) );
	  
	  // Get the VISA IO interface type that is going to be used.
      viCheckErr( viGetAttribute( io, VI_ATTR_INTF_TYPE, &interfaceType ) );

	  // Setup the VISA IO interface to report the IO_COMPLETION event.
	  viCheckErr( viEnableEvent( io, VI_EVENT_IO_COMPLETION, VI_QUEUE, VI_NULL));

      /* If port is serial, then configure properly for RS232 communications */
      if( interfaceType == VI_INTF_ASRL )  
      {
      	 ViInt32 uiDataRate;
      	 
		 user_if = RS232_IF;
         // Configure VISA IO Serial communication to use termination character for reading.
         viCheckErr( viSetAttribute( io, VI_ATTR_ASRL_END_IN, 
                                     VI_ASRL_END_TERMCHAR ) );

         // Configure VISA IO Serial communication to use termination character for writing.                                     
       	 viCheckErr( viSetAttribute( io, VI_ATTR_ASRL_END_OUT, 
                                     VI_ASRL_END_TERMCHAR ) );

		
	  	 // At the present moment the first channel will be the master channel.
   	  	 AmetekDC_GetAttributeViInt32(vi, VI_NULL, AMETEKDC_ATTR_DATARATE, &uiDataRate);

         // Configure the data rate of the serial connection.
   	     viCheckErr( viSetAttribute( io, VI_ATTR_ASRL_BAUD, 
   	                                 uiDataRate));                                     
      }		 
#ifdef __AMETEKDC_SPECIFIC_ADDRESS_HANDLING__	  
      else if( interfaceType == VI_INTF_GPIB )  
      {
      	  user_if = GPIB_IF;
		  // Behaviour is to check to see if a secondary address was specified, if non has
      	  // been specified then we use the primary address as the secondary. Since the 
      	  // master unit it assumed. If the use specifies a specific secondary address the 
      	  // driver should initially use this one.
      	  viCheckErr( viGetAttribute( io, VI_ATTR_GPIB_SECONDARY_ADDR, &busAddress) );

          if( busAddress == VI_NO_SEC_ADDR)
          {
        	// Since no secondary address was specified communication with the 
        	// master unit is desire by the user.
        	//
      	  	// We know that the bus address and the GPIB address are the same.  So we 
	      	// need to make both addresses the same by loading the GPIB address here.
      		viCheckErr( viGetAttribute( io, VI_ATTR_GPIB_PRIMARY_ADDR, &busAddress) );
          }
	
	  	  // Convert the bus address into a string for storage as the default address.
	  	  viSPrintf(io, rdBuffer, "%u", busAddress);    	  

          // Set the Bus address to the GPIB address or the default Bus Address
          // for XG. This overrides the MCHADDR configuration option.  
          AmetekDC_SetAttributeViString(vi, VI_NULL, AMETEKDC_ATTR_BUS_ADDRESS, rdBuffer );	 							  
      }
	  else if (interfaceType == VI_INTF_TCPIP)
	  {
	      user_if = ENET_IF;
			
		  // NIMAX says these are all settable attributes:
		  viCheckErr(viSetAttribute (io, VI_ATTR_TMO_VALUE, 5000));
		  //viCheckErr(viSetAttribute (io, VI_ATTR_MAX_QUEUE_LENGTH, 50));
		  viCheckErr(viSetAttribute (io, VI_ATTR_SEND_END_EN, VI_TRUE));
		  viCheckErr(viSetAttribute (io, VI_ATTR_TERMCHAR_EN, VI_TRUE));
		  viCheckErr(viSetAttribute (io, VI_ATTR_DMA_ALLOW_EN, VI_FALSE));
		  viCheckErr(viSetAttribute (io, VI_ATTR_FILE_APPEND_EN, VI_FALSE));
		  viCheckErr(viSetAttribute (io, VI_ATTR_TCPIP_KEEPALIVE, VI_FALSE));

		  viCheckErr(viSetAttribute (io, VI_ATTR_TERMCHAR, '\r')); 
	
	      // Bonus attemtp to set other attributes not deemed settable by NIMAX but might help:
      	  viCheckErr(viSetAttribute (io, VI_ATTR_TCPIP_NODELAY, VI_TRUE ));
	      viCheckErr(viSetAttribute (io, VI_ATTR_IO_PROT, VI_PROT_NORMAL ));
      	  viCheckErr(viSetAttribute (io, VI_ATTR_SUPPRESS_END_EN, VI_TRUE ));
      	  viCheckErr(viSetAttribute (io, VI_ATTR_WR_BUF_OPER_MODE, VI_FLUSH_WHEN_FULL ));
    	  viCheckErr(viSetAttribute (io, VI_ATTR_USER_DATA, 0x00000000));  
	
		}
		
#endif //#ifdef __AMETEKDC_SPECIFIC_ADDRESS_HANDLING__	  		  
   }/* end if( !Ivi_Simulating( vi ) ) */
   
   /* Add attributes */
   checkErr( AmetekDC_InitAttributes( vi, modelType, IDQuery ) );
   
   /* Link the channels with the appropriate range tables */
   checkErr( AmetekDC_SetRangeTableAddress( vi, channelString ) );

    // Clear any errors.
   checkErr( AmetekDC_ClearStatus(  vi, AMETEKDC_MASTER_CHANNEL ));    
   checkErr( AmetekDC_reset( vi ) );
   
Error:
   if( error < VI_SUCCESS )
   {
      if( !Ivi_Simulating( vi ) && io )
      {      
         viClose( io );
      }
   }
   Ivi_UnlockSession( vi, VI_NULL );    
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_close                                                           
 * Purpose:  This function closes the instrument.                            
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 * Note:  This function must unlock the session before calling
 *           Ivi_Dispose.
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_close( ViSession vi )
{
   ViStatus error = VI_SUCCESS;
      
   checkErr( AmetekDC_IviClose( vi ) );

Error:    
   Ivi_Dispose( vi );  
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_IviClose                                                        
 * Purpose:  This function performs all of the drivers clean-up operations   
 *           except for closing the IVI session.  This function is called by 
 *           AmetekDC_close or by an IVI class driver. 
 *
 *           Note:  This function must close the I/O session and set 
 *           IVI_ATTR_IO_SESSION to 0.
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_IviClose( ViSession vi )
{
   ViStatus  error = VI_SUCCESS;
   ViSession io    = VI_NULL;

  checkErr( Ivi_LockSession( vi, VI_NULL ) );
  
   /* Do not lock here.  The caller manages the lock. */
   checkErr( Ivi_GetAttributeViSession( vi, VI_NULL, IVI_ATTR_IO_SESSION, 
                                        0, &io ) );

Error:
   Ivi_SetAttributeViSession( vi, VI_NULL, IVI_ATTR_IO_SESSION, 0, VI_NULL );
   if( io )                                                      
   {
      viClose( io );
   }
   Ivi_UnlockSession( vi, VI_NULL );   
   return( error );   
}




/***************************************************************************** 
 *------------------ Configuration/Output functions -------------------------*
 *****************************************************************************/
/***************************************************************************** 
 *--------------------- Basic Operation functions ---------------------------*
 *****************************************************************************/

/*****************************************************************************
 * Function: AmetekDC_ConfigureOutputEnabled                                   
 * Purpose:  This function controls whether or not the signal the power supply
 *           generates is present at the output terminal.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ConfigureOutputEnabled( ViSession vi,
                                                 ViConstString channelName,
                                                 ViBoolean enabled )
{
   ViStatus error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
    
   viCheckParm( Ivi_SetAttributeViBoolean( vi, channelName, 
                                           AMETEKDC_ATTR_OUTPUT_ENABLED,
                                           0, enabled ), 3, "Enabled" );
                                          
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}  

/*****************************************************************************
 * Function: AmetekDC_ConfigureOutputRange                                   
 * Purpose:  This function determines the output range in which the power
 *           supply operates.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ConfigureOutputRange( ViSession vi, 
                                               ViConstString channelName,
                                               ViInt32 rangeType,
                                               ViReal64 range )
{
   ViStatus error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
    
   if( Ivi_RangeChecking( vi ) )
   {
      viCheckParm( Ivi_CoerceChannelName( vi, channelName, VI_NULL), 2, 
                                          "Channel Name");
        
      if( ( rangeType != AMETEKDC_VAL_RANGE_CURRENT ) && 
          ( rangeType != AMETEKDC_VAL_RANGE_VOLTAGE ) )
      {
         viCheckParm( IVI_ERROR_INVALID_PARAMETER, 3, 
                      "Range Type" );
      }
   }

   if( rangeType == AMETEKDC_VAL_RANGE_CURRENT )
   {
      /* Verify the range value against the current limit range table */
      viCheckParm( Ivi_CheckAttributeViReal64( 
                   vi, channelName, AMETEKDC_ATTR_CURRENT_LIMIT, 0, range ), 4, 
                   "Invalid Current value for 'Range' parameter "
                   "in AmetekDC_ConfigureOutputRange." );
   }
   else
   {
      /* Verify the range value against the voltage level range table */
      viCheckParm( Ivi_CheckAttributeViReal64( 
                   vi, channelName, AMETEKDC_ATTR_VOLTAGE_LEVEL, 0, range ), 4, 
                   "Invalid Voltage value for 'Range' parameter "
                   "in AmetekDC_ConfigureOutputRange." );
   }
    
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_ConfigureVoltageLevel                                   
 * Purpose:  This function configures the voltage level the power supply
 *           attempts to generate.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ConfigureVoltageLevel( ViSession vi,
                                                ViConstString channelName,
                                                ViReal64 level )
{
   ViStatus error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
    
   viCheckParm( Ivi_SetAttributeViReal64( vi, channelName,
                                          AMETEKDC_ATTR_VOLTAGE_LEVEL,
                                          0, level ), 3, "Voltage Level" );
                                          
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );                                         
}

/*****************************************************************************
 * Function: AmetekDC_ConfigureCurrentLimit                                   
 * Purpose:  This function configures the level of the current limit.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 * NOTE: The Current limit must be set before sending a voltage level to the
 *           power supply upon initialization.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ConfigureCurrentLimit( ViSession vi,
                                                ViConstString channelName,
                                                ViInt32 behavior,
                                                ViReal64 limit )
{
   ViStatus error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   // By setting this attribute first the Ocp Limit will be updated 
   // according to the behaviour specified.
   viCheckParm( Ivi_SetAttributeViReal64( vi, channelName, 
                                          AMETEKDC_ATTR_CURRENT_LIMIT, 0, limit ), 
                                          4, "Current Limit" );
   
   viCheckParm( Ivi_SetAttributeViInt32( vi, channelName, 
                                         AMETEKDC_ATTR_CURRENT_LIMIT_BEHAVIOR, 
                                         0, behavior ),
                                         3, "Current Limit Behavior" );
                                          

   
   // Set the Over Current Protection Limit if Trip is selected 
   // Management of this done through the behavior setting attribute callbacks.

   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_ConfigureFrontPanelTimeout                                   
 * Purpose:  This function configures the amount of time that the settings will be
 *              displayed with no activity before the XG returns to normal display mode.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ConfigureFrontPanelTimeout( ViSession vi,
                                                ViConstString channelName,
                                                ViInt32 timeout)
{
   ViStatus error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   // By setting this attribute first the Ocp Limit will be updated 
   // according to the behaviour specified.
   viCheckParm( Ivi_SetAttributeViInt32( vi, channelName, 
                                          AMETEKDC_ATTR_FP_INACTIVITY_TIMEOUT, 0, timeout ), 
                                          4, "Front Panel Inactivity Timeout" );

   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_QueryFrontPanelTimeout                                   
 * Purpose:  This function queries the amount of time that the settings will be
 *              displayed with no activity before the XG returns to normal display mode.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryFrontPanelTimeout( ViSession vi,
                                                ViConstString channelName,
                                                ViInt32* timeout)
{
   ViStatus error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   // By setting this attribute first the Ocp Limit will be updated 
   // according to the behaviour specified.
   viCheckParm( Ivi_GetAttributeViInt32( vi, channelName, 
                                          AMETEKDC_ATTR_FP_INACTIVITY_TIMEOUT, 0, timeout ), 
                                          4, "Front Panel Inactivity Timeout" );

   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}                                                

/***************************************************************************** 
 *---------------- Protection Mechanism functions ---------------------------*
 *****************************************************************************/

/*****************************************************************************
 * Function: AmetekDC_ConfigureOVP                                   
 * Purpose:  This function configures the over-voltage protection limit
 *           and enables or disables the OVP 
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ConfigureOVP( ViSession vi,
                                       ViConstString channelName,
                                       ViBoolean enabled,
                                       ViReal64 limit )
{
   ViStatus error = VI_SUCCESS;
  
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

  viCheckParm( Ivi_SetAttributeViReal64( vi, channelName, 
                                             AMETEKDC_ATTR_OVP_LIMIT,
                                             0, limit ), 4, "OVP Limit" );

  viCheckParm( Ivi_SetAttributeViBoolean( vi, channelName, 
                                             AMETEKDC_ATTR_OVP_ENABLED,
                                             0, enabled ), 3, "OVP Limit" );
  
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_ConfigureProtectionSetpoint                                   
 * Purpose:  This function configures the protection limits for Voltage and
 *           Current. ie. Over Current Protection, Under Current 
 *           Protection, etc. 
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ConfigureProtectionSetpoint( ViSession vi,
                                                      ViConstString channelName,
                                                      ViAttr protectionType,
                                                      ViBoolean behavior, 
                                                      ViReal64 limit )
{
   ViStatus error = VI_SUCCESS;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );


   switch(protectionType)
   {
      case AMETEKDC_ATTR_OVP_LIMIT:
           checkErr( AmetekDC_ConfigureOVP( vi, channelName, behavior, limit ) );
           break;
		   
      case AMETEKDC_ATTR_OCP_LIMIT:
           checkErr( AmetekDC_ConfigureOCP( vi, channelName, behavior, limit ) );
           break;
		   
      case AMETEKDC_ATTR_UVP_LIMIT:
           checkErr( AmetekDC_ConfigureUVP( vi, channelName, behavior, limit ) );
           break;
		   		   
	default:
           viCheckParm( ( AMETEKDC_ERROR_INVALID_PARAMETER ), 3,
                     "Protection Type parameter is invalid\n" );
	    break;
   }
   
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_ConfigureProtectionLatch                                 
 * Purpose:  This function sets the output response in the event of an a  
 *           protection tripping.  The output is either latched off, or can be 
 *           set to auto recover once the  condition has been corrected.
 *
 *           Currently only the AC Fail condition, or an Over Temperature 
 *           condition are latchable. 
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ConfigureProtectionLatch( ViSession vi,
                                                   ViConstString channelName,
                                                   ViAttr protectionType, 
                                                   ViBoolean state )
{
   ViStatus    error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   if( Ivi_RangeChecking( vi ) )
   {
      if( ( protectionType != AMETEKDC_ATTR_AC_FAIL_RESPONSE ) && 
          ( protectionType != AMETEKDC_ATTR_OVER_TEMPERATURE_RESPONSE ) &&
          ( protectionType != AMETEKDC_ATTR_OVER_TEMPERATURE_SLAVE_RESPONSE ) &&
          ( protectionType != AMETEKDC_ATTR_FOLDBACK_RESPONSE))
      {
         Ivi_SetErrorInfo( vi, VI_FALSE, AMETEKDC_ERROR_INVALID_PARAMETER, 0,
                   "The value for the Protection Type parameter is invalid\n"
                   "for the AmetekDC_ConfigureProtectionLatch function." );
                    
         viCheckErr( AMETEKDC_ERROR_INVALID_PARAMETER );
      }
      
   }

   viCheckErr( Ivi_SetAttributeViBoolean( vi, channelName, protectionType, 0, 
                                          state ) );

   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
      
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

static IviRangeTableEntry attrFoldbackSettingRangeTableEntries[] =
	{
		{AMETEKDC_VAL_FOLD_NONE, 0, 0, "NONE", 0},
		{AMETEKDC_VAL_FOLD_CC, 0, 0, "CC", 0},
		{AMETEKDC_VAL_FOLD_CV, 0, 0, "CV", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable attrFoldbackSettingRangeTable =
	{
		IVI_VAL_DISCRETE,
        VI_FALSE,
        VI_FALSE,
        VI_NULL,
        attrFoldbackSettingRangeTableEntries,
	};

/*****************************************************************************
 * Function: AmetekDC_ConfigureFoldbackSetting                  
 * Purpose:   This function disables the power supply output when the output 
 *            enters the fold condition
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ConfigureFoldbackSetting( ViSession vi, 
                                                   ViConstString channelName,
                                                   ViInt32 foldMode )
{
   ViStatus error = VI_SUCCESS;
   ViString cmd;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
    
   /* obtain commmand string from range table */
   viCheckParm( Ivi_GetViInt32EntryFromValue( foldMode,
                                              &attrFoldbackSettingRangeTable, 
                                              VI_NULL, VI_NULL, VI_NULL, 
                                              VI_NULL, &cmd, VI_NULL ), 3,
                                              "Fold Mode" );
    
   checkErr( Ivi_SetAttributeViString( vi, channelName, 
                                       AMETEKDC_ATTR_FOLDBACK_SETTING,
                                       0, cmd ) );
                                    
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );                                  
                              
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_ConfigureFoldDelay                                   
 * Purpose:  This function sets a time delay before foldback protection triggers.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ConfigureFoldDelay( ViSession vi, 
                                             ViConstString channelName,
                                             ViReal64 value )
{
   ViStatus    error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_SetAttributeViReal64( vi,channelName, 
                                          AMETEKDC_ATTR_FOLD_DELAY,
                                          0, value ), 3, "Fold Delay Time" );
                                       
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );                                  
                              
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );                                       
}  

/*****************************************************************************
 * Function: AmetekDC_SetProtectionMask                                   
 * Purpose:  This function allows the masking and unmasking of the XGs 
 *              protection alarms.  Not all alarms can be masked so see the 
 *              user manual for details on which ones can be masked.  The 
 *              input parameter is a bit masked made up of the PRT_teFLAG
 *              enum type logically ORed together. 
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SetProtectionMask( ViSession vi, 
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean *value,
                                             ViUInt16 flagPosition )
{
   ViStatus    error = VI_SUCCESS;
   ViUInt32    protMaskReg = 0;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   // Get current latch value.
   viCheckParm( Ivi_GetAttributeViInt32( vi,channelName, 
                                          AMETEKDC_ATTR_PROTECTION_MASK,
                                          0, &protMaskReg ), 4, "Protection Mask" );
 

   // Update the latch according to the value 
   if( value == VI_FALSE)
   {
   	protMaskReg &= ~(1<<flagPosition);
   }
   else
   {
   	protMaskReg |= (1<<flagPosition);
   }
   
   
   viCheckParm( Ivi_SetAttributeViInt32( vi,channelName, 
                                          AMETEKDC_ATTR_PROTECTION_MASK,
                                          0, protMaskReg ), 4, "Protection Mask" );
                                       
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );                                  
                              
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );                                       
}  

/***************************************************************************** 
 *------------ Monitoring Power Supply Status functions ---------------------*
 *****************************************************************************/

/***************************************************************************** 
 *---------------- Configuration/Set,Get,Check functions --------------------*
 *****************************************************************************/

/*****************************************************************************
 * Function: AmetekDC_GetAttribute<type> Functions                                    
 * Purpose:  These functions enable the instrument driver user to get 
 *           attribute values directly. There are typesafe versions for 
 *           ViInt32, ViReal64, ViString, ViBoolean, and ViSession attributes.     
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_GetAttributeViInt32( ViSession vi, 
                                              ViConstString channelName, 
                                              ViAttr attributeId, 
                                              ViInt32 *value )
{                                                                                                           
   return( Ivi_GetAttributeViInt32( vi, channelName, attributeId, 
                                   IVI_VAL_DIRECT_USER_CALL, 
                                   value ) );
}                                           

ViStatus _VI_FUNC AmetekDC_GetAttributeViReal64( ViSession vi, 
                                              ViConstString channelName, 
                                              ViAttr attributeId, 
                                              ViReal64 *value )
{                                                                                                           
   return( Ivi_GetAttributeViReal64( vi, channelName, attributeId, 
                                    IVI_VAL_DIRECT_USER_CALL, 
                                    value ) );
}

ViStatus _VI_FUNC AmetekDC_GetAttributeViString( ViSession vi, 
                                               ViConstString channelName, 
                                               ViAttr attributeId, 
                                               ViInt32 bufSize, 
                                               ViChar value[ ] ) 
{   
   return( Ivi_GetAttributeViString( vi, channelName, attributeId, 
                                    IVI_VAL_DIRECT_USER_CALL, 
                                    bufSize, value ) );
}

ViStatus _VI_FUNC AmetekDC_GetAttributeViBoolean( ViSession vi, 
                                                ViConstString channelName, 
                                                ViAttr attributeId, 
                                                ViBoolean *value )
{                                                                                                           
   return( Ivi_GetAttributeViBoolean( vi, channelName, attributeId, 
                                     IVI_VAL_DIRECT_USER_CALL, 
                                     value ) );
}                                          

ViStatus _VI_FUNC AmetekDC_GetAttributeViSession( ViSession vi, 
                                                ViConstString channelName, 
                                                ViAttr attributeId, 
                                                ViSession *value )
{                                                                                                           
   return( Ivi_GetAttributeViSession( vi, channelName, attributeId, 
                                     IVI_VAL_DIRECT_USER_CALL, 
                                     value ) );
}                                                                                                           

/*****************************************************************************
 * Function: AmetekDC_SetAttribute<type> Functions                                    
 * Purpose:  These functions enable the instrument driver user to set 
 *           attribute values directly.  There are typesafe versions for 
 *           ViInt32, ViReal64, ViString, ViBoolean, and ViSession datatypes.     
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SetAttributeViInt32( ViSession vi, 
                                              ViConstString channelName, 
                                              ViAttr attributeId, 
                                              ViInt32 value )
{                                                                                                           
   return( Ivi_SetAttributeViInt32( vi, channelName, attributeId, 
                                   IVI_VAL_DIRECT_USER_CALL, 
                                   value ) );
}              

ViStatus _VI_FUNC AmetekDC_SetAttributeViReal64( ViSession vi, 
                                               ViConstString channelName, 
                                               ViAttr attributeId, 
                                               ViReal64 value )
{                                                                                                           
   return( Ivi_SetAttributeViReal64( vi, channelName, attributeId, 
                                    IVI_VAL_DIRECT_USER_CALL, 
                                    value ) );
}

ViStatus _VI_FUNC AmetekDC_SetAttributeViString( ViSession vi, 
                                               ViConstString channelName, 
                                               ViAttr attributeId, 
                                               ViConstString value ) 
{   
   return( Ivi_SetAttributeViString( vi, channelName, attributeId, 
                                    IVI_VAL_DIRECT_USER_CALL, 
                                    value ) );
}

ViStatus _VI_FUNC AmetekDC_SetAttributeViBoolean( ViSession vi, 
                                                ViConstString channelName, 
                                                ViAttr attributeId, 
                                                ViBoolean value )
{                                                                                                           
   return( Ivi_SetAttributeViBoolean( vi, channelName, attributeId, 
                                     IVI_VAL_DIRECT_USER_CALL, 
                                     value ) );
}

ViStatus _VI_FUNC AmetekDC_SetAttributeViSession( ViSession vi, 
                                                ViConstString channelName, 
                                                ViAttr attributeId, 
                                                ViSession value )
{                                                                                                           
   return( Ivi_SetAttributeViSession( vi, channelName, attributeId, 
                                     IVI_VAL_DIRECT_USER_CALL, 
                                     value ) );
}                                                                                                           

/*****************************************************************************
 * Function: AmetekDC_CheckAttribute<type> Functions                                  
 * Purpose:  These functions enable the instrument driver user to check  
 *           attribute values directly.  These functions check the value you
 *           specify even if you set the AMETEKDC_ATTR_RANGE_CHECK 
 *           attribute to VI_FALSE.  There are typesafe versions for ViInt32, 
 *           ViReal64, ViString, ViBoolean, and ViSession datatypes.   
 *
 * NOTE: Recommended functions for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_CheckAttributeViInt32( ViSession vi, 
                                                ViConstString channelName, 
                                                ViAttr attributeId, 
                                                ViInt32 value )
{                                                                                                           
   return( Ivi_CheckAttributeViInt32( vi, channelName, attributeId, 
                                     IVI_VAL_DIRECT_USER_CALL, 
                                     value ) );
}

ViStatus _VI_FUNC AmetekDC_CheckAttributeViReal64( ViSession vi, 
                                                 ViConstString channelName, 
                                                 ViAttr attributeId, 
                                                 ViReal64 value )
{                                                                                                           
   return( Ivi_CheckAttributeViReal64( vi, channelName, attributeId, 
                                      IVI_VAL_DIRECT_USER_CALL, 
                                      value ) );
}

ViStatus _VI_FUNC AmetekDC_CheckAttributeViString( ViSession vi, 
                                                 ViConstString channelName, 
                                                 ViAttr attributeId, 
                                                 ViConstString value )  
{   
   return( Ivi_CheckAttributeViString( vi, channelName, attributeId, 
                                      IVI_VAL_DIRECT_USER_CALL, 
                                      value ) );
}

ViStatus _VI_FUNC AmetekDC_CheckAttributeViBoolean( ViSession vi, 
                                                  ViConstString channelName, 
                                                  ViAttr attributeId, 
                                                  ViBoolean value )
{                                                                                                           
   return( Ivi_CheckAttributeViBoolean( vi, channelName, attributeId, 
                                       IVI_VAL_DIRECT_USER_CALL, 
                                       value ) );
}

ViStatus _VI_FUNC AmetekDC_CheckAttributeViSession( ViSession vi, 
                                                  ViConstString channelName, 
                                                  ViAttr attributeId, 
                                                  ViSession value )
{                                                                                                           
   return( Ivi_CheckAttributeViSession( vi, channelName, attributeId, 
                                       IVI_VAL_DIRECT_USER_CALL, 
                                       value ) );
}



 
/***************************************************************************** 
 *----------------------- Action/Status functions ---------------------------*
 *****************************************************************************/ 
/***************************************************************************** 
 *---------------------- General Action functions ---------------------------*
 *****************************************************************************/
 

/*****************************************************************************
 *------------------------ Measurement Range Table --------------------------*
 *****************************************************************************/

/****************************************************************************
 * Function: AmetekDC_Measure
 * Purpose:  This function measures either the current or voltage that the
 *           power supply is generating or the input signals from the APG system.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_Measure( ViSession vi,
                                  ViConstString channelName,
                                  ViInt32 measurementType,
                                  ViReal64 *measurement )
{
                               
   ViStatus       error = VI_SUCCESS;
   ViReal64       value = 0.0;
   ViInt32		 cmdIdx = 0;
   ViChar         cmdString[ SMALL_BUFF_SIZE ];
   ViString       partialCmd;
   ViAttr         attrId;
   ViConstString  coercedChannel;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
    
   IviDCPwr_InterchangeCheck( vi, "AmetekDC_Measure" );
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );
   
   if( measurement == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 4, 
                   "Null value for Measurement parameter." );
   }
                                                 
   if( !Ivi_Simulating( vi ) )
   {
      ViSession io = Ivi_IOSession( vi );
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );
      
	// Send the command.
	switch(measurementType)
	{
		case AMETEKDC_VAL_MEASURE_CURRENT:
		    cmdIdx = AMETEKDC_VAL_MEAS_CURR;
		    break;

		case AMETEKDC_VAL_MEASURE_VOLTAGE:
		    cmdIdx = AMETEKDC_VAL_MEAS_VOLT;
		    break;

		case AMETEKDC_VAL_MEASURE_ANALOG_CURR_INP:
		    cmdIdx = AMETEKDC_VAL_MEAS_CAP;      
		    break;
			
		case AMETEKDC_VAL_MEASURE_ANALOG_VOLT_INP:
		    cmdIdx = AMETEKDC_VAL_MEAS_VAP;
		    break;

		case AMETEKDC_VAL_MEASURE_ISOL_ANALOG_CURR_INP:
		    cmdIdx = AMETEKDC_VAL_MEAS_CAP_ISOL;      
		    break;

		case AMETEKDC_VAL_MEASURE_ISOL_ANALOG_VOLT_INP:
		    cmdIdx = AMETEKDC_VAL_MEAS_VAP_ISOL;
		    break;
			
		default:			
       	    viCheckParm(IVI_ERROR_INVALID_PARAMETER, 3, "Measure Type out of range");			
		    break;	
	}
	
  	checkErr( AmetekDC_SendCmd(vi, io, channelName, cmdIdx, "?") );	
      viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_INP_FORMAT, &value ) );
   }
  
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:    
   if( measurement != VI_NULL )
   {
      *measurement = value;
   }
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );  
}

/*****************************************************************************
 * Function: AmetekDC_SetRegister                                   
 * Purpose:  This function sets the value of one of the SCPI status registers. 
 *             To specify a status register use the AMETEKDC_VAL_<register name>
 * 		   defined in the header file.    
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SetRegister( ViSession vi, 
                                      ViConstString channelName,
                                      ViInt32 setRegister, 
                                      ViInt16 setValue )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   // Verify that the command index specified by the setRegister parameter
   // does fall in the range of register indices.
   if(( setRegister < AMETEKDC_VAL_FIRST_REGISTER_CMD) || 
   	( setRegister > AMETEKDC_VAL_LAST_REGISTER_CMD))
   {
   	viCheckParm(IVI_ERROR_INVALID_PARAMETER, 3, "Set Register out of range");
   }

   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );
                                              
   if( !Ivi_Simulating( vi ) )
   {
      ViSession io = Ivi_IOSession( vi );
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

	// Send the command.
      checkErr( AmetekDC_SendInt16Cmd(vi, io, channelName, setRegister, setValue) );
      
   }

   	
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_ClearStatus                                   
 * Purpose:  This function will clear the status data structures.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ClearStatus( ViSession vi, ViConstString channelName )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );
   
   if( !Ivi_Simulating( vi ) )              
   {
      ViSession io = Ivi_IOSession( vi ); 
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

	// Send the command.
      checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_STAT_CLEAR, "") );
	     
   }

   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_ResetOutputProtection                                   
 * Purpose:  This function will clear all protection mechanisms and re-enable
 *           output. If the condition that caused the alarm still exists, the 
 *           protection will trip again.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ResetOutputProtection( ViSession vi,
                                                ViConstString channelName )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                      "Channel Name" );
  
   if( !Ivi_Simulating( vi ) )
   {
      ViSession io = Ivi_IOSession( vi );
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

	// Send the command.
      checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_PROT_CLEAR, "") );
	
   }
    
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
    
Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
} 


/***************************************************************************** 
 *---------------------- General Query functions ---------------------------*
 *****************************************************************************/
 



/*****************************************************************************
 * Function: AmetekDC_QueryOutputState                                   
 * Purpose:  This function returns a value indicating if the power supply
 *           is in a particular output state.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryOutputState( ViSession vi, 
                                           ViConstString channelName, 
                                           ViInt32 outputState, 
                                           ViBoolean *inState )
{
   ViStatus       error    = VI_SUCCESS;
   ViBoolean      value    = VI_FALSE;
   ViInt32        mask;
   ViInt32        cmdIdx;
   ViInt32        registerValue = 0;
   ViConstString  coercedChannel;
   
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ),
                                       2, "Incorrect Channel Name" );
                                       
   if( inState == VI_NULL )
   {
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 2, 
                   "Null value for 'In State' parameter of "
                   "AmetekDC_QueryOutputState." );
   }

   // The reason for using a switch table instead of a lookup table is to 
   // make changing the command string easier so that we have lower 
   // maintenance costs moving forward as well as making this driver 
   // easier to adapted to any future products.
   switch(outputState)
   {
   	case AMETEKDC_VAL_OUTPUT_CONSTANT_VOLTAGE:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_COND;
		mask = AMETEKDC_MASK_CV;
		break;
		
	case AMETEKDC_VAL_OUTPUT_CONSTANT_CURRENT:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_COND;
		mask = AMETEKDC_MASK_CC;
		break;
		
	case AMETEKDC_VAL_OUTPUT_OVER_VOLTAGE:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND;		
		mask = AMETEKDC_MASK_OVP;
		break;
		
	case AMETEKDC_VAL_OUTPUT_UNDER_VOLTAGE:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND;		
		mask = AMETEKDC_MASK_UVP;
		break;
		
	case AMETEKDC_VAL_OUTPUT_OVER_CURRENT:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND;		
		mask = AMETEKDC_MASK_OCP;
		break;
		
	case AMETEKDC_VAL_OUTPUT_OVER_TEMPERATURE:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND;		
		mask = AMETEKDC_MASK_OTP;
		break;
		
	default: 
		// Throw error as this is in invalid query.
		viCheckErr( IVI_ERROR_INVALID_PARAMETER );		
		break; 
   }
   
   if( !Ivi_Simulating( vi ) )
   {
      ViSession io = Ivi_IOSession( vi );
        
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

	// Query the register that indicates the output state.
      checkErr( AmetekDC_SendCmd(vi, io, coercedChannel, cmdIdx, "?") );
      viCheckErr( viScanf(io, AMETEKDC_SHORT_PARAM_INP_FORMAT, &registerValue ) );
   }

   // Determine the output state based on the command.
   if( ( registerValue & mask ) != 0 )
   {
   	value = VI_TRUE;
   }
   
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:
   if( inState != NULL )
   {
      *inState = value;
   }
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

 
/*****************************************************************************
 * Function: AmetekDC_QueryRegister                                   
 * Purpose:  This function returns the value of one of the SCPI status registers. 
 *             To specify a status register use the AMETEKDC_VAL_<register name>
 * 		   defined in the header file.    
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryRegister( ViSession vi,
                                        ViConstString channelName,
                                        ViInt32 queryRegister, 
                                        ViInt16 *registerValue )
{
   ViStatus       error = VI_SUCCESS;
   ViString       partialCmd;
   ViChar         cmdString[ SMALL_BUFF_SIZE ];
   ViConstString  coercedChannel;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   // Verify that the command index specified by the setRegister parameter
   // does fall in the range of register indices.
   if(( queryRegister < AMETEKDC_VAL_FIRST_REGISTER_CMD) || 
   	( queryRegister > AMETEKDC_VAL_LAST_REGISTER_CMD))
   {
   	viCheckParm(IVI_ERROR_INVALID_PARAMETER, 3, "Set Register out of range");
   }
   
   // Validate the channel.
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );   

   // Check for null pointer.
   if( registerValue == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 4, 
                   "Null value for Register Value parameter." );
   }
                                              
   if( !Ivi_Simulating( vi ) )
   {
      ViSession io = Ivi_IOSession( vi );
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );
      
	// Send the command.
      checkErr( AmetekDC_SendCmd(vi, io, channelName, queryRegister, "?") );
      viCheckErr( viScanf(io, AMETEKDC_SHORT_PARAM_INP_FORMAT, registerValue ) );
   }
   else
   {
      *registerValue = 0;
   }
    
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_QueryMaxCurrentLimit                  
 * Purpose:  This function returns the maximum programmable current limit
 *           that the power supply accepts.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryMaxCurrentLimit( ViSession vi,
                                               ViConstString channelName,
                                               ViReal64 VoltageLevel,
                                               ViReal64 *maxCurrentLimit )
{
   ViStatus          error = VI_SUCCESS;
   IviRangeTablePtr  currentLimitRangeTable = VI_NULL;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) ); 

   if( maxCurrentLimit == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 4, 
                   "Null value for maxCurrentLimit." );
   }

   checkErr( Ivi_GetAttrRangeTable( vi, channelName, AMETEKDC_ATTR_CURRENT_LIMIT,
                                    &currentLimitRangeTable) );
   
   viCheckErr( Ivi_GetViReal64EntryFromIndex( 0, currentLimitRangeTable, 
                                              VI_NULL, maxCurrentLimit, 
                                              VI_NULL, VI_NULL, VI_NULL ) );
   
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


 /*****************************************************************************
 * Function: AmetekDC_QueryMaxVoltageLevel                  
 * Purpose:  This function returns the maximum programmable voltage level
 *           that the power supply accepts.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryMaxVoltageLevel( ViSession vi,
                                               ViConstString channelName,
                                               ViReal64 CurrentLimit,
                                               ViReal64 *maxVoltageLevel )
{
   ViStatus          error = VI_SUCCESS;
   IviRangeTablePtr  voltageLevelRangeTable = VI_NULL;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   if( maxVoltageLevel == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 4, 
                   "Null value for maxVoltageLevel." );
   }
   
   checkErr( Ivi_GetAttrRangeTable( vi, channelName, AMETEKDC_ATTR_VOLTAGE_LEVEL,
                                    &voltageLevelRangeTable) );

   viCheckErr( Ivi_GetViReal64EntryFromIndex( 0, voltageLevelRangeTable, 
                                              VI_NULL, maxVoltageLevel, 
                                              VI_NULL, VI_NULL, VI_NULL ) );

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_QuerySetpoint                                   
 * Purpose:  This function returns the setpoint value for a particular
 *           output parameter. ie. Voltage, Current, and the Fold Delay setting( seconds )
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QuerySetpoint( ViSession vi, 
                                        ViConstString channelName,
                                        ViAttr attrId, 
                                        ViReal64 *setpointLevel )
{
   ViStatus error = VI_SUCCESS;
   ViReal64 value = 0.0;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) ); 

   if( setpointLevel == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 4, 
                   "Null value for Setpoint Level parameter." );
   }

   checkErr( Ivi_GetAttributeViReal64( vi, channelName, attrId, 0, &value ) );
   
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
   
Error:
   if( setpointLevel != NULL )
   {
      *setpointLevel = value;
   }
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_QueryProtectionSettings                                   
 * Purpose:  This function returns the setpoint value and behavior for a particular
 *           output protection parameter. ie. Over-Current Protection.
 *           The valid query parameters are OVP,OCP, and  UVP
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryProtectionSettings( ViSession vi,
                                                  ViConstString channelName,
                                                  ViAttr protectionType,
                                                  ViBoolean *behavior, 
                                                  ViReal64 *level )
{
   ViStatus error = VI_SUCCESS;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) ); 
   checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );
   
   if( behavior == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 4, 
                   "Null value for Behavior parameter." );
   }
   
   if( level == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 5, 
                   "Null value for Level parameter." );
   }
   
   if( protectionType == AMETEKDC_ATTR_OVP_LIMIT )
   {
      checkErr( Ivi_GetAttributeViReal64( vi, channelName, 
                                          AMETEKDC_ATTR_OVP_LIMIT, 0, level ) );
	  
	*behavior = AMETEKDC_VAL_PROTECTION_TRIP;
   }
   else if( protectionType == AMETEKDC_ATTR_OCP_LIMIT )
   {
      checkErr( Ivi_GetAttributeViReal64( vi, channelName, 
                                          AMETEKDC_ATTR_OCP_LIMIT, 0, level ) );
                                          
      checkErr( Ivi_GetAttributeViBoolean( vi, channelName, 
                                           AMETEKDC_ATTR_CURRENT_LIMIT_BEHAVIOR, 
                                           0, behavior ) );
   }
   else if( protectionType == AMETEKDC_ATTR_UVP_LIMIT )
   {
      checkErr( Ivi_GetAttributeViReal64( vi, channelName, 
                                          AMETEKDC_ATTR_UVP_LIMIT, 0, level ) );
                                          
      checkErr( Ivi_GetAttributeViBoolean( vi, channelName, 
                                           AMETEKDC_ATTR_UVP_BEHAVIOR, 
                                           0, behavior ) );
   }
   else
   {
      Ivi_SetErrorInfo( vi, VI_FALSE, AMETEKDC_ERROR_INVALID_PARAMETER, 0,
                    "The value for the Protection Type parameter is invalid\n"
                    "for the AmetekDC_QueryOutputSettings function." );
                    
      viCheckErr( AMETEKDC_ERROR_INVALID_PARAMETER );
   }
      
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_QueryProtectionMask                                   
 * Purpose:  This function allows the user to query the XGs protection 
 *              alarms bit mask.  The response is a flag register which is 
 *              defined by the PRT_teFLAG enums.  The enums indicate bit
 *              positions.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryProtectionMask( ViSession vi, 
                                             ViConstString channelName,
                                             ViAttr protectionType,                                              
                                             ViUInt32* value )                                          
{
   ViStatus    error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
  
   viCheckParm( Ivi_GetAttributeViInt32( vi,channelName, 
                                          AMETEKDC_ATTR_PROTECTION_MASK,
                                          0, value ), 3, "Fold Delay Time" );

   
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );                                  
                              
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );                                       
}  


/*****************************************************************************
 * Function: AmetekDC_QueryProtectionLatch                                 
 * Purpose:  This function queries the output response in the event of an 
 *           AC Fail condition, or an Over Temperature condition. The output
 *           is either latched off, or can be set to auto recover once the 
 *           condition has been corrected.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryProtectionLatch( ViSession vi,
                                               ViConstString channelName,
                                               ViAttr protectionType, 
                                               ViBoolean *state )
{
   ViStatus error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
    
   checkErr( Ivi_GetAttributeViBoolean( vi, channelName, protectionType, 
                                        0, state ) );
                                          
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_QueryProtectionTripped                                   
 * Purpose:  This function returns whether a protection mechanism has tripped.  It interogates the
 *           condition register to determine the state.  If event detection is required then  this should
 *           be done through the QueryRegister API call.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryProtectionTripped( ViSession vi,
                                                 ViConstString channelName,
                                                 ViInt32 protectionType, 
                                                 ViBoolean *value )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;
   ViInt32        cmdIdx = 0;
   ViInt32       mask = 0;
   ViInt32 	registerValue;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );
                                       
   if( value == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 4, 
                   "Null value for Value parameter." );
   }


   // The reason for using a switch table instead of a lookup table is to 
   // make changing the command string easier so that we have lower 
   // maintenance costs moving forward as well as making this driver 
   // easier to adapted to any future products.
   switch(protectionType)
   {		
	case AMETEKDC_VAL_OVP:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND;		
		mask = AMETEKDC_MASK_OVP;
		break;
		
	case AMETEKDC_VAL_UVP:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND;		
		mask = AMETEKDC_MASK_UVP;
		break;
		
	case AMETEKDC_VAL_OCP:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND;		
		mask = AMETEKDC_MASK_OCP;
		break;
		
	case AMETEKDC_VAL_OVER_TEMP:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND;		
		mask = AMETEKDC_MASK_OTP;
		break;

	case AMETEKDC_VAL_FOLDBACK:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND;		
		mask = AMETEKDC_MASK_OTP;
		break;
		
	case AMETEKDC_VAL_AC_FAIL:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND;		
		mask = AMETEKDC_MASK_OTP;
		break;
		
	case AMETEKDC_VAL_FAN:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_PROT_COND;		
		mask = AMETEKDC_MASK_FAN;
		break;

	case AMETEKDC_VAL_SD:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_COND;		
		mask = AMETEKDC_MASK_SD;
		break;

	case AMETEKDC_VAL_INTERLOCK:
		cmdIdx = AMETEKDC_VAL_STAT_OPER_SHUT_COND;		
		mask = AMETEKDC_MASK_INT;
		break;
	
	default: 
		// Throw error as this is in invalid query.
		viCheckErr( IVI_ERROR_INVALID_PARAMETER );
		break; 
   }  
   
   if( !Ivi_Simulating( vi ) )
   {
      ViSession io = Ivi_IOSession( vi );
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

	// Query the register that indicates the output state.
      checkErr( AmetekDC_SendCmd(vi, io, coercedChannel, cmdIdx, "?") );
      viCheckErr( viScanf(io, AMETEKDC_SHORT_PARAM_INP_FORMAT, &registerValue ) );

	// Check the condition register to see if the protection is tripped.
	if( (registerValue & mask) != 0)
	{
		*value = VI_TRUE;	
	}
	
   } 
   else
   {
      /* for Simulating */
      *value = VI_FALSE;
   }
    
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
    
Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_QueryEnabled                                   
 * Purpose:  This function returns the enabled value for a particular
 *           setting. The valid query types are Output: Disabled/Enabled, 
 *           Output at Power-on: Disabled/Enabled.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryEnabled( ViSession vi, ViConstString channelName,
                                       ViAttr attrId, ViBoolean *enabled )
{
   ViStatus   error = VI_SUCCESS;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   if( enabled == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 4, 
                   "Null value for Enabled parameter." );
   }
   
   switch( attrId )
   {
      case AMETEKDC_ATTR_OUTPUT_ENABLED:
      case AMETEKDC_ATTR_OUTPUT_ENABLED_POWER_ON:
         viCheckErr( Ivi_GetAttributeViBoolean( vi, channelName, attrId, 0, 
                                                enabled ) );
         break;
         
      default:
         Ivi_SetErrorInfo( vi, VI_FALSE, AMETEKDC_ERROR_INVALID_PARAMETER, 0,
                        "The attribute Id for the Query parameter is invalid\n"
                        "for the AmetekDC_QueryEnabled function." );
                    
         viCheckErr( AMETEKDC_ERROR_INVALID_PARAMETER );
         break;
   }
      
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );   
}
    
/*****************************************************************************
 * Function: AmetekDC_QueryAddress                                   
 * Purpose:  This function returns the address of the instruments currently "selected" 
 *           (multichannel address).
 *
 * Note:     This is not a multichannel function.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryAddress( ViSession vi, 
                                       ViInt32 *address )
{
   #define ADDRESS_STR_SIZE 16
   ViStatus error = VI_SUCCESS;
   ViChar viStrAddress[ADDRESS_STR_SIZE];  
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   //  Get the string value of the current Bus Address.
   checkErr( Ivi_GetAttributeViString( vi, VI_NULL, AMETEKDC_ATTR_BUS_ADDRESS, VI_NULL, ADDRESS_STR_SIZE, viStrAddress ) );

   // Convert into an integer value.
   *address = atoi((char*) viStrAddress);

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_QuerySource                                   
 * Purpose:  This function returns the Source for one of the following 
 *           attributes: Remote.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QuerySource( ViSession vi, ViConstString channelName, 
                                      ViAttr attrId, ViChar source[] )
{
   ViStatus error = VI_SUCCESS;
   ViInt32  trigSource;   /* Used to hold the integer value of trig source */

   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   if( source == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 3, 
                   "Null value for Source parameter." );
   }
   
   switch( attrId )
   {
      case AMETEKDC_ATTR_REMOTE_SOURCE_CURR:
      case AMETEKDC_ATTR_REMOTE_SOURCE_VOLT:
         viCheckErr( Ivi_GetAttributeViString( vi, channelName, attrId, 0, 
                                               256, source ) );
         break;
         
      default:
         Ivi_SetErrorInfo( vi, VI_FALSE, AMETEKDC_ERROR_INVALID_PARAMETER, 0,
                   "The attribute Id value for the Query parameter is invalid\n"
                   "for the AmetekDC_QuerySource function." );
                   
         viCheckErr( AMETEKDC_ERROR_INVALID_PARAMETER );
         break;
   }
   
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
      
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_QueryMode                                  
 * Purpose:  This function returns the Mode for one of the following 
 *           attributes: Remote mode, Current share mode, analog programming
 *           CAP and VAP modes,  and foldback mode.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryMode( ViSession vi, ViConstString channelName,
                                    ViAttr attrId, ViChar mode[] )
{
   ViStatus error = VI_SUCCESS;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   if( mode == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 3, 
                   "Null value for Mode parameter." );
   }
   
   switch( attrId )
   {
      case AMETEKDC_ATTR_FOLDBACK_SETTING: 
      case AMETEKDC_ATTR_CURRENT_SHARE_MODE: 
      case AMETEKDC_ATTR_REMOTE_SOURCE_VOLT:
      case AMETEKDC_ATTR_REMOTE_SOURCE_CURR:	  	
      case AMETEKDC_ATTR_REMOTE_MODE:
         viCheckErr( Ivi_GetAttributeViString( vi, channelName, attrId, 0, 
                                               256, mode ) );
         break;
         
      default:
         Ivi_SetErrorInfo( vi, VI_FALSE, AMETEKDC_ERROR_INVALID_PARAMETER, 0,
                  "The attribute Id value for the Query parameter is invalid\n"
                  "for the AmetekDC_QueryMode function." );
                  
         viCheckErr( AMETEKDC_ERROR_INVALID_PARAMETER );
         break;
   }
      
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
      
Error:                           
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

 
/***************************************************************************** 
 *--------------------- Auto Sequence Range Tables --------------------------*
 *****************************************************************************/

 
static IviRangeTableEntry autoSequenceRepeatRangeTableEntries[] =
{
   {0, 9999, 0, "", 0},
   {IVI_RANGE_TABLE_LAST_ENTRY}
};

static IviRangeTable autoSequenceRepeatRangeTable =
{
   IVI_VAL_RANGED,
   VI_TRUE,
   VI_TRUE,
   VI_NULL,
   autoSequenceRepeatRangeTableEntries,
};

static IviRangeTableEntry autoSequenceStateRangeTableEntries[] =
{
   {AMETEKDC_VAL_RUN,   0, 0, "RUN",  0},
   {AMETEKDC_VAL_PAUSE, 0, 0, "PAUS", 0},
   {AMETEKDC_VAL_STOP,  0, 0, "STOP", 0},
   {IVI_RANGE_TABLE_LAST_ENTRY}
};

static IviRangeTable autoSequenceStateRangeTable =
{
   IVI_VAL_DISCRETE,
   VI_FALSE,
   VI_FALSE,
   VI_NULL,
   autoSequenceStateRangeTableEntries,
};

/***************************************************************************** 
 *------------------- Auto Sequence Action functions ------------------------*
 *****************************************************************************/

/***************************************************************************** 
 *-------------------- Programming Sequences functions ----------------------*
 *****************************************************************************/


/*****************************************************************************
 * Function: AmetekDC_SetAutoSequenceRecordStart                                                      
 * Purpose:  This function sets the XG's that receive the message into 
 *           the auto sequence record state.  All subsequent commands
 *           will be recorded into the XG but no action will be taken till
 *           the auto sequence program is executed.  To bring the XG 
 *           back into normal mode send the SetAutoSequenceRecordStop
 *           command.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SetAutoSequenceRecordStart( 
                                                   ViSession vi,
                                                   ViConstString channelName )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 
                                       2, "Channel Name" );

    if( !Ivi_Simulating( vi ) )              
   {
      ViSession io = Ivi_IOSession( vi ); 
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );
      
      checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_PROG_START_RECORDING, "") );               
   }
       												 
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_SetAutoSequenceRecordStop                                                      
 * Purpose:  This function sets the XG's that receive the message into 
 *           the auto sequence record stop.  The XG will nolonger record
 *           the commands sent to it into the Auto sequence buffer. 
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SetAutoSequenceRecordStop( 
                                                   ViSession vi,
                                                   ViConstString channelName )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 
                                       2, "Channel Name" );

    if( !Ivi_Simulating( vi ) )              
   {
      ViSession io = Ivi_IOSession( vi ); 
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );
      
      checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_PROG_STOP_RECORDING, "") );               
   }
       
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
      
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/***************************************************************************** 
 *------------------------- Deleting functions ------------------------------*
 *****************************************************************************/
 
/*****************************************************************************
 * Function: AmetekDC_DeleteAllSequences                                                         
 * Purpose:  This function deletes all of the programmed Auto Sequences stored
 *           in the instruments memory.  
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_DeleteAllSequences( ViSession vi, 
                                             ViConstString channelName )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );
                                       
   if( !Ivi_Simulating( vi ) )
   {
      ViSession io = Ivi_IOSession( vi ); 
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

	// Send the command.
      checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_PROG_DELETE_ALL, "") );

   }
       
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
      
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

   
/***************************************************************************** 
 *-------------------- Programming Sequences functions ----------------------*
 *****************************************************************************/


/*****************************************************************************
 * Function: AmetekDC_SetAutoSequenceRepeat                                                      
 * Purpose:  This function determines the number of times a particular sequence
 *           will be executed. ie. Once, Twice... Infinite.  To set the repeat
 *           count to infinite you should pass a null in the repeat count.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SetAutoSequenceRepeat( 
                                                   ViSession vi,
                                                   ViConstString channelName,
                                                   ViInt16 repeatCount )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 
                                       2, "Channel Name" );

   if( Ivi_RangeChecking( vi ) )
   {
      
      /* Check the sequence count to ensure it is between 0 and 9999 */                                                 
      viCheckParm( Ivi_GetViInt32EntryFromValue( repeatCount, 
                                                 &autoSequenceRepeatRangeTable,
                                                 VI_NULL, VI_NULL, VI_NULL, 
                                                 VI_NULL, VI_NULL, VI_NULL ), 3,
                                                 "Sequence Count" );
   }

   if( !Ivi_Simulating( vi ) )              
   {
      ViSession io = Ivi_IOSession( vi ); 
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );
      
      if( repeatCount > 0 )
      {
      	   // Send repeat count.
         checkErr( AmetekDC_SendInt16Cmd(vi, io, channelName, AMETEKDC_VAL_PROG_REPEAT, repeatCount) );            
      }
      else
      {
      	   // Use infinite parameter. Send repeat count.
         checkErr( AmetekDC_SendStringCmd(vi, io, channelName, AMETEKDC_VAL_PROG_REPEAT, "INF") );         
      } 
      
   }
       
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
      
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}



/***************************************************************************** 
 *---------------------- Running Sequence functions ------------------------*
 *****************************************************************************/
 

/*****************************************************************************
 * Function: AmetekDC_SendAutoSequenceSoftwareTrigger                                                      
 * Purpose:  This function sends the ":INIT:IMM" command, which will instruct 
 *           the power supply to jump to execute the auto sequence program that
 *           is currently in memory.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SendAutoSequenceSoftwareTrigger(
                                                ViSession vi,
                                                ViConstString channelName )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );
    
   if( !Ivi_Simulating( vi ) )
   {
      ViSession io = Ivi_IOSession( vi );
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );


	// Send the command.
      checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_INITIATE_IMM, "") );

   }
        
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
         
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}



/***************************************************************************** 
 *------------------- Query Auto Sequence functions -------------------------*
 *****************************************************************************/
 
/*****************************************************************************
 * Function: AmetekDC_QueryAutoSequenceState
 * Purpose:  This queries the state of the selected auto sequence. It will 
 *           either be in run, pause, or stop mode.   
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryAutoSequenceState( ViSession vi,
                                                     ViConstString channelName,
                                                     ViChar state[] )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );

   if( state == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 3, 
                   "Null value for State." );
   }

   if( !Ivi_Simulating( vi ) )              
   {
      ViSession io = Ivi_IOSession( vi ); 
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );
      
	// Use infinite parameter. Send repeat count.
      checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_PROG_STATE, "?") );
      viCheckErr( viScanf( io, AMETEKDC_STRING_PARAM_INP_FORMAT, state ) ); 
	          
   }
       
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
      
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_QuerySpecificSequenceRepeat                                                       
 * Purpose:  This function queries the supply to determine the number of times
 *           a specific auto sequence will execute.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryAutoSequenceRepeat( ViSession vi,
                                                      ViConstString channelName,
                                                      ViChar repeatCount[] )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );

 
   if( repeatCount == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 4, 
                   "Null value for Sequence Count parameter." );
   }

   if( !Ivi_Simulating( vi ) )              
   {
      ViSession io = Ivi_IOSession( vi ); 
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

	// Use infinite parameter. Send repeat count.
      checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_PROG_REPEAT, "?") );
      viCheckErr( viScanf( io, AMETEKDC_STRING_PARAM_INP_FORMAT, repeatCount ) ); 
        
   }
      
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
      
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_QueryAutoSequenceDwellTime                                  
 * Purpose:  This function sets the time delay used after each viPrintf() has
 *           executed. 
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryAutoSequenceDwellTime( ViSession vi, 
                                      ViConstString channelName,
                                      ViInt32* delayTime )
{
   ViStatus error = VI_SUCCESS;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   checkErr( Ivi_GetAttributeViInt32( vi, channelName,
                                       AMETEKDC_ATTR_AUTO_SEQUENCE_DWELL_TIME,
                                       0, delayTime ) );

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/***************************************************************************** 
 *--------------------- Utility functions(Exported ) ------------------------*
 *****************************************************************************/
 
/*****************************************************************************
 * Function: AmetekDC_SetAutoSequenceState                                  
 * Purpose:  This function changes the state of the auto sequence program in 
 *           memory.  There are three possible states that can be set: RUN,
 *           STOP, and PAUSe. 
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SetAutoSequenceState (ViSession vi,
                                      ViConstString channelName,
                                      ViChar state)
{
  ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;
   ViString         cmdParam;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );
   
   // Get the parameter.
   viCheckParm( Ivi_GetViInt32EntryFromValue( state, 
                                              &autoSequenceStateRangeTable,
                                              VI_NULL, VI_NULL, VI_NULL, 
                                              VI_NULL, &cmdParam, VI_NULL ), 
                                              3, "Auto Sequence State" );
	
   if( !Ivi_Simulating( vi ) )
   {
      ViSession io = Ivi_IOSession( vi ); 
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

	       
	  // Send state.
      checkErr( AmetekDC_SendStringCmd(vi, io, coercedChannel, AMETEKDC_VAL_PROG_STATE, cmdParam) );
   }
       
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
      
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_SetAutoSequenceDwellTime                                  
 * Purpose:  This function sets the time delay used after step of an auto sequence program.
 *               The parameter is in the units of seconds.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SetAutoSequenceDwellTime( ViSession vi, 
															   ViConstString channelName,
															   ViInt32 delayTimeSec )
{
   ViStatus error = VI_SUCCESS;
   ViConstString  coercedChannel;
   ViString         cmdParam;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );

   if( Ivi_RangeChecking( vi ) )
   {
      
      /* Check the sequence count to ensure it is in range */                                                 
      viCheckParm( Ivi_GetViInt32EntryFromValue( delayTimeSec, 
                                                 &attrAutoSequenceDwellTimeRangeTable,
                                                 VI_NULL, VI_NULL, VI_NULL, 
                                                 VI_NULL, VI_NULL, VI_NULL ), 2, "Delay Time" );
   }
      
   checkErr( Ivi_SetAttributeViInt32( vi, coercedChannel,
                                       AMETEKDC_ATTR_AUTO_SEQUENCE_DWELL_TIME,
                                       0, delayTimeSec ) );

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}



/*****************************************************************************
 * Function: AmetekDC_SaveUserSettings                                  
 * Purpose:  This function saves all of the current instrument settings into 
 *           the specified User Setting Location.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SaveUserSettings( ViSession vi,
                                           ViConstString channelName,
                                           ViInt16 userSetting)
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );     
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );

   if( Ivi_RangeChecking( vi ) )
   {
      if( ( userSetting < 1 ) || ( userSetting > AMETEKDC_NUMB_USER_SETTINGS ) )
      {
         viCheckParm( ( IVI_ERROR_INVALID_PARAMETER ), 3, "User Setting" );
      }
      
   }
   
   if( !Ivi_Simulating( vi ) )                
   {
      ViSession io = Ivi_IOSession( vi ); 
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

	  
	// Send the command.
      checkErr( AmetekDC_SendInt16Cmd(vi, io, channelName, AMETEKDC_VAL_SYST_SAVE, userSetting) );
   }
   
   // The XG performs a blocking operation during a save which requires  this additional
   // delay before the unit is read to receive the next command.
   Delay(AMETEKDC_SAVE_DELAY_SEC);   
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/***************************************************************************** 
 *----------------------- System Setup functions ----------------------------*
 *****************************************************************************/
 
/*****************************************************************************
 * Function: AmetekDC_SetOutputEnabledPowerOn                                   
 * Purpose:  This function controls whether or not the power supply powers on
 *           with the output enabled.
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SetOutputEnabledPowerOn( ViSession vi,
                                                  ViConstString channelName,
                                                  ViBoolean enabled )
{
   ViStatus error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
    
   checkErr( Ivi_SetAttributeViBoolean( vi, channelName, 
                                        AMETEKDC_ATTR_OUTPUT_ENABLED_POWER_ON,
                                        0, enabled ) );
                                          
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_SetCurrentShareMode                      
 * Purpose:  These functions enable the instrument driver user to  
 *           get or clear the error information the driver associates with the
 *           IVI session.   
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SetCurrentShareMode( ViSession vi,
                                              ViConstString channelName,
                                              ViConstString mode )
{
   ViStatus error = VI_SUCCESS;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   if( Ivi_RangeChecking( vi ) )
   {
      if( ( strcmp( mode, "MAST" ) != 0 ) &&
          ( strcmp( mode, "SLAV" ) != 0 ) )
      {
         viCheckParm( ( AMETEKDC_ERROR_INVALID_PARAMETER ), 3, 
                        "Current Share Mode" );
      }
   
   }
   
   checkErr( Ivi_SetAttributeViString( vi, channelName, 
                                       AMETEKDC_ATTR_CURRENT_SHARE_MODE,
                                       0, mode ) );
   
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_SetCommandTimeDelay                                  
 * Purpose:  This function sets the time delay used after each command that has
 *           been sent to the attached supply.  
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SetCommandTimeDelay( ViSession vi, ViReal64 delayTimeSec )
{
   ViStatus error = VI_SUCCESS;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   // Make sure the delay time falls into the range table.
   if( Ivi_RangeChecking( vi ) )
   {
      
      viCheckParm( Ivi_GetViReal64EntryFromValue( delayTimeSec, 
                                                 &attrConfigurableDelayRangeTable,
                                                 VI_NULL, VI_NULL, VI_NULL, 
                                                 VI_NULL, VI_NULL, VI_NULL ), 2, "Delay Time" );
   }   
   checkErr( Ivi_SetAttributeViReal64( vi, VI_NULL,
                                       AMETEKDC_ATTR_CONFIGURABLE_DELAY,
                                       0, delayTimeSec ) );

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_RecallUserSetting                                  
 * Purpose:  This function loads one of the stored user settings which determines
 *           the new settings for the instrument.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_RecallUserSetting( ViSession vi,
                                            ViConstString channelName,
                                            ViInt16 userSetting)
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );     

   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );
   
   if( Ivi_RangeChecking( vi ) )
   {
      if( ( userSetting < 1 ) || ( userSetting > AMETEKDC_NUMB_USER_SETTINGS ) )
      {
         viCheckParm( ( AMETEKDC_ERROR_INVALID_PARAMETER ), 3, "User Setting" );
      }
      
   }
   
   if( !Ivi_Simulating( vi ) )                
   {
      ViSession io = Ivi_IOSession( vi ); 
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

	// Send the command.
      checkErr( AmetekDC_SendInt16Cmd(vi, io, channelName, AMETEKDC_VAL_SYST_REC, userSetting) );
      
   }
   
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


static IviRangeTableEntry attrRemoteModeRangeTableEntries[] =
	{
		{AMETEKDC_VAL_LOC, 0, 0, "LOC", 0},
		{AMETEKDC_VAL_REM, 0, 0, "REM", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable attrRemoteModeRangeTable =
	{
		IVI_VAL_DISCRETE,
        VI_FALSE,
        VI_FALSE,
        VI_NULL,
        attrRemoteModeRangeTableEntries,
	};

/*****************************************************************************
 * Function: AmetekDC_SetRemoteMode                       
 * Purpose:  This function selects the mode that the power supply runs in.
 *           The options are Local mode and Remote mode.  When in remote mode
 *           the front panel is locked, in local the front panel is unlocked.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SetRemoteMode( ViSession vi,
                                        ViConstString channelName,
                                        ViInt32 state )
{
   ViStatus error = VI_SUCCESS;
   ViString cmd;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   /* Get the command string from the range table */
   viCheckParm( Ivi_GetViInt32EntryFromValue( state, &attrRemoteModeRangeTable,
                                              VI_NULL, VI_NULL, VI_NULL, 
                                              VI_NULL, &cmd, VI_NULL ), 3, 
                                              "Remote State" );
   
   checkErr( Ivi_SetAttributeViString( vi, channelName, 
                                       AMETEKDC_ATTR_REMOTE_MODE,
                                       0, cmd ) );
                                       
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

static IviRangeTableEntry attrRemoteSourceRangeTableEntries[] =
	{
		{AMETEKDC_VAL_LOC, 0, 0, "LOC", 0},
		{AMETEKDC_VAL_IAV, 0, 0, "IAV", 0},
		{AMETEKDC_VAL_IAR, 0, 0, "IAR", 0},
		{AMETEKDC_VAL_AVOL, 0, 0, "AVOL", 0},
		{AMETEKDC_VAL_ARES, 0, 0, "ARES", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable attrRemoteSourceRangeTable =
	{
		IVI_VAL_DISCRETE,
        VI_FALSE,
        VI_FALSE,
        VI_NULL,
        attrRemoteSourceRangeTableEntries,
	};

/*****************************************************************************
 * Function: AmetekDC_SelectVoltRemoteControlSource                       
 * Purpose:  This function provides the API call to select the source to control
 *           voltage set point for VAP.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SelectVoltRemoteControlSource( ViSession vi,
                                                    ViConstString channelName,
                                                    ViInt32 source )
{
   ViStatus error = VI_SUCCESS;
   ViString cmd;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
    /* Get the command string from the range table */
   viCheckParm( Ivi_GetViInt32EntryFromValue( source, 
                                              &attrRemoteSourceRangeTable,
                                              VI_NULL, VI_NULL, VI_NULL, 
                                              VI_NULL, &cmd, VI_NULL ), 3, 
                                              "Source" );
   
   checkErr( Ivi_SetAttributeViString( vi, channelName, 
                                       AMETEKDC_ATTR_REMOTE_SOURCE_VOLT,
                                       0, cmd ) );
                                       
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_SelectCurrRemoteControlSource                       
 * Purpose:  This function provides the API call to select the source to control
 *           current set point for CAP.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SelectCurrRemoteControlSource( ViSession vi,
                                                    ViConstString channelName,
                                                    ViInt32 source )
{
   ViStatus error = VI_SUCCESS;
   ViString cmd;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
    /* Get the command string from the range table */
   viCheckParm( Ivi_GetViInt32EntryFromValue( source, 
                                              &attrRemoteSourceRangeTable,
                                              VI_NULL, VI_NULL, VI_NULL, 
                                              VI_NULL, &cmd, VI_NULL ), 3, 
                                              "Source" );
   
   checkErr( Ivi_SetAttributeViString( vi, channelName, 
                                       AMETEKDC_ATTR_REMOTE_SOURCE_CURR,
                                       0, cmd ) );
                                       
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_SelectVoltAnalogProgrammingRange                       
 * Purpose:  This function provides the API call to select the input range for
 *           for VAP.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SelectVoltAnalogProgrammingRange( 
                                                ViSession vi,
                                                ViConstString channelName,
                                                ViReal64 range )
{
   ViStatus error = VI_SUCCESS;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_SetAttributeViReal64( vi, channelName, 
                                         AMETEKDC_ATTR_VOLT_ANALOG_PROGRAMMING_RANGE,
                                         0, range ), 2, "APR Range" );
                                     
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
   
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_SelectCurrAnalogProgrammingRange                       
 * Purpose:  This function provides the API call to select the input range for
 *           for CAP.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SelectCurrAnalogProgrammingRange( 
                                                ViSession vi,
                                                ViConstString channelName,
                                                ViReal64 range )
{
   ViStatus error = VI_SUCCESS;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   viCheckParm( Ivi_SetAttributeViReal64( vi, channelName, 
                                         AMETEKDC_ATTR_CURR_ANALOG_PROGRAMMING_RANGE,
                                         0, range ), 2, "APR Range" );
                                     
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );
   
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
/*---------------- Ivi_DCPwr Required Utility Functions ---------------------*
 *****************************************************************************/
 
 /*****************************************************************************
 * Function: AmetekDC_self_test                       
 * Purpose:  This function causes the power supply to run a self diagnostic test.
 *           It returns a '0' if the test passed, or non zero to indicate a problem.
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_self_test( ViSession vi, ViInt16 *selfTestResult,
                                    ViChar selfTestMessage[],
                                    ViConstString channelName)
{
   ViStatus error = VI_SUCCESS;
   ViBoolean isValid = VI_FALSE;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );     

   if( selfTestResult == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 2, 
                   "Null value for 'Self Test Result' parameter." );
   }
   
   if( selfTestMessage == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 3, 
                   "Null value for 'Self Test Message' parameter." );
   }
   
   if( !Ivi_Simulating( vi ) )                
   {
      ViSession   io = Ivi_IOSession( vi ); 
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

      // Need to verify the channel string.
      if(strcmp(AMETEKDC_MASTER_CHANNEL, channelName) != 0)
      	{
      	     // Master channel not specified. Check multichannel address 
      	     // (RS-485 bus address)
 	     AmetekDC_VerifyChannelName(vi, channelName, &isValid);
      	     if( isValid == VI_FALSE)
           {
         		// The channel address is incorrect. Thow an error.
	      	    	viCheckParm( IVI_ERROR_INVALID_PARAMETER, 4, 
      	             "Value outside of range for 'Channel Name' parameter." ); 	
      	     }
      }
	 
      // Send the command.
     checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SYST_TEST, "?") );
     viCheckErr( viScanf( io, AMETEKDC_SHORT_PARAM_INP_FORMAT, selfTestResult ) ); 
   	
      if( selfTestResult == 0)
      {
         strcpy( selfTestMessage, "Passed" );
      }
      else
      {
         strcpy( selfTestMessage, "Failed" );
      }
      
   }
   else
   {
      *selfTestResult = 0;  /* Simulate Self Test */
      strcpy( selfTestMessage, "Passed" );
   }
       
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:
    Ivi_UnlockSession( vi, VI_NULL );
    return( error );
}


/*****************************************************************************
 * Function: AmetekDC_Preset
 * Purpose:  This function resets the instrument and applies default settings
 *           from the IVI Configuration Store based on the logical name
 *           from which the session was created.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_Preset( ViSession vi, ViConstString channelName  )
{
   ViStatus error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   if( !Ivi_Simulating( vi ) )                
   {
      ViSession   io = Ivi_IOSession( vi ); 
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );   
	  
      checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_STAT_PRESET, ""));
   }
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_reset                                                         
 * Purpose:  This function resets the instrument. 
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_reset( ViSession vi )
{
   ViStatus    error = VI_SUCCESS;

   
   checkErr( AmetekDC_ResetMultichannelPowerSupply(vi, AMETEKDC_MASTER_CHANNEL) );

Error:
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_resetMultichannelPowerSupply                                                         
 * Purpose:  This function resets a multichannel instrument.    
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ResetMultichannelPowerSupply( 
                                                ViSession vi,
                                                ViConstString channelName )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   if(strcmp(channelName, AMETEKDC_MASTER_CHANNEL))
   {
       viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2, 
                                       "Channel Name" );
   }
   else
   {
      coercedChannel = AMETEKDC_MASTER_CHANNEL;
   }
   	
   if( !Ivi_Simulating( vi ) )              
   {
      ViSession   io = Ivi_IOSession( vi ); /* call only when locked */
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

	// Send the command.
      checkErr( AmetekDC_SendCmd(vi, io, coercedChannel, AMETEKDC_VAL_SYST_RES, "") );      
   }
   
   checkErr( AmetekDC_DefaultInstrSetup( vi ) );                                
   Delay(AMETEKDC_RESET_DELAY_SEC);
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );                                      

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_ResetWithDefaults
 * Purpose:  This function resets the instrument and applies default settings
 *           from the IVI Configuration Store based on the logical name
 *           from which the session was created.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ResetWithDefaults( ViSession vi )
{
   ViStatus error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   checkErr( AmetekDC_reset( vi ) );
   checkErr( Ivi_ApplyDefaultSetup( vi ) );
   
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/**************************************************************************** 
 *  Function: AmetekDC_Disable
 *  Purpose:  This function places the instrument in a quiescent state as 
 *            quickly as possible. May be similar but not the same as reset.
 *            For the XG implementation the programing is disabled and the
 *            output turned off in a compound statment which should get the
 *            fastest possible response time without reseting the device.
 ****************************************************************************/
ViStatus _VI_FUNC AmetekDC_Disable( ViSession vi )
{
   ViStatus error = VI_SUCCESS;
   ViString outpCmdStr;
   ViString progStateCmdStr;
   ViString progStateParamStr;
   ViChar disableCmdStr [BUFFER_SIZE];
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
    
   if( !Ivi_Simulating( vi ) ) 
   {
      ViSession   io = Ivi_IOSession( vi ); /* call only when locked */
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

	// Get the Output State command from the command string range table.
	viCheckErr( Ivi_GetViInt32EntryFromValue( AMETEKDC_VAL_OUTP_STATE, 
                                              &scpiCmdStringRangeTable,
                                              VI_NULL, VI_NULL, VI_NULL, 
                                              VI_NULL, &outpCmdStr, VI_NULL));

	// Get the Program State command from the command string range table.
	viCheckErr( Ivi_GetViInt32EntryFromValue( AMETEKDC_VAL_PROG_STATE, 
                                              &scpiCmdStringRangeTable,
                                              VI_NULL, VI_NULL, VI_NULL, 
                                              VI_NULL, &progStateCmdStr, VI_NULL ));	

   // Get the auto sequence program stop state parameter.
   viCheckErr( Ivi_GetViInt32EntryFromValue( AMETEKDC_VAL_STOP, 
                                              &autoSequenceStateRangeTable,
                                              VI_NULL, VI_NULL, VI_NULL, 
                                              VI_NULL, &progStateParamStr, VI_NULL ));
   
	viSPrintf(io, disableCmdStr, "%s %s;%s %s", 
		      outpCmdStr,
		      "OFF", 
		      progStateCmdStr,
		      progStateParamStr);

	// Send the command to turn off the output with broadcast address
	viCheckErr( viPrintf( io, 
		      disableCmdStr, 
		      AMETEKDC_BROADCAST_CHANNEL, 
		      AMETEKDC_BROADCAST_CHANNEL) ); 

      // Wait for required delay before sending the next command.
      checkErr( AmetekDC_Delay( vi ) ); /* User-Configurable delay */  

   }

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_revision_query                                                  
 * Purpose:  This function returns the driver and instrument revisions.   
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_revision_query( ViSession vi, ViChar driverRev[], 
                                         ViChar instrRev[] )
{
   ViStatus    error = VI_SUCCESS;
    
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   if( driverRev == VI_NULL )
   {
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 2, 
                   "Null address for Driver Revision" );
   }
    
   if( instrRev == VI_NULL )
   {
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 3, 
                   "Null address for Instrument Revision" );
   }

   checkErr( Ivi_GetAttributeViString( vi, VI_NULL, 
                                       AMETEKDC_ATTR_SPECIFIC_DRIVER_REVISION, 
                                       0, 256, driverRev ) );  
                                       
   checkErr( Ivi_GetAttributeViString( vi, VI_NULL, 
                                       AMETEKDC_ATTR_INSTRUMENT_FIRMWARE_REVISION, 
                                       0, 256, instrRev ) );
                                       
   checkErr( AmetekDC_CheckStatus( vi, AMETEKDC_MASTER_CHANNEL ) );

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}



/*****************************************************************************
 *------------------- Error Functions ---------------------------------------* 
 *****************************************************************************/

/*****************************************************************************
 * Function: AmetekDC_QueryErrorCount                                                    
 * Purpose:  This function queries the attached instrument or any multichannel instrument for the number
 * of errors in its queue and returns the result.    
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_QueryErrorCount( ViSession vi, 
                                                 ViConstString channelName,
                                                 ViInt32 *errCount)
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );   
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2,
                                       "Channel Name" );

   if( error == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 2, 
                   "Null value for Error Code parameter." );
   }
   
   if( !Ivi_Simulating( vi ) )
   {
      ViSession   io = Ivi_IOSession( vi ); /* call only when locked */
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

      checkErr( AmetekDC_SendCmd( vi, io, channelName, AMETEKDC_VAL_SYST_ERROR_COUNT, "?" ));	   
      viCheckErr( viScanf( io, AMETEKDC_USHORT_PARAM_INP_FORMAT, errCount ) );	       
   }
   else
   {
       /* Simulate Error Query */
       *errCount = 0;
   }
      
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
} 

/*****************************************************************************
 * Function: AmetekDC_error_query                                                     
 * Purpose:  This function queries the instrument error queue and returns   
 *           the result.    
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_error_query( ViSession vi, ViInt32 *errCode,
                                      ViChar errorMessage[] )
{
   return AmetekDC_ErrorQueryMultichannel( vi, AMETEKDC_MASTER_CHANNEL, errCode, errorMessage);   
}

/*****************************************************************************
 * Function: AmetekDC_ErrorQueryMultichannel                                                    
 * Purpose:  This function queries a multichannel instrument error queue and returns   
 *           the result.    
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ErrorQueryMultichannel( ViSession vi, 
                                                 ViConstString channelName,
                                                 ViInt32 *errCode, 
                                                 ViChar errorMessage[] )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );   
   
   viCheckParm( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ), 2,
                                       "Channel Name" );

   if( errCode == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 2, 
                   "Null value for Error Code parameter." );
   }
   
   if( errorMessage == VI_NULL )
   {   
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 3, 
                   "Null value for Error Message parameter." );
   }
   
   if( !Ivi_Simulating( vi ) )
   {
      ViSession   io = Ivi_IOSession( vi ); /* call only when locked */
      checkErr( Ivi_SetNeedToCheckStatus( vi, VI_TRUE ) );

      checkErr( AmetekDC_SendCmd( vi, io, channelName, AMETEKDC_VAL_SYST_ERROR, "?" ));	   
      viCheckErr( viScanf( io, "%ld,%256[^\n]", errCode, errorMessage ) );	       
   }
   else
   {
       /* Simulate Error Query */
       *errCode = 0;
       strcpy( errorMessage, "No error" );
   }
      
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_error_message                                                  
 * Purpose:  This function translates the error codes returned by this       
 *           instrument driver into user-readable strings.  
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 * NOTE: The caller can pass VI_NULL for the vi parameter. This 
 *           is useful if one of the init functions fail.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_error_message( ViSession vi, ViStatus errorCode,
                                        ViChar errorMessage[256] )
{
   ViStatus    error = VI_SUCCESS;
   
   static IviStringValueTable errorTable = 
   {
      {AMETEKDC_ERROR_TRIGGER_NOT_SOFTWARE, "Trigger source is not software trigger."},
      {AMETEKDC_ERROR_INVALID_PARAMETER,    "One of the function parameters is invalid."},
      IVIDCPWR_ERROR_CODES_AND_MSGS,
      {VI_NULL,            VI_NULL},
   };
        
   if( vi )
   {
      checkErr( Ivi_LockSession( vi, VI_NULL ) );
   }

   /* all VISA and IVI error codes are handled as well as codes in the table */
   if( errorMessage == VI_NULL )
   {
      viCheckParm( IVI_ERROR_INVALID_PARAMETER, 3, 
                   "Null address for Error Message" );
   }

   checkErr( Ivi_GetSpecificDriverStatusDesc( vi, errorCode, errorMessage, 
                                              errorTable ) ); 
    
Error:
   if( vi )
   {
       Ivi_UnlockSession( vi, VI_NULL );
   }
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_ClearError Function                       
 * Purpose:  This functions enables the instrument driver user to clear the  
 *              error information the driver associates with the IVI session.
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ClearError( ViSession vi )                                                        
{                                                                                                           
   return( Ivi_ClearErrorInfo( vi ) );                                                                             
}

/*****************************************************************************
 * Function: AmetekDC_ClearErrorInfo Function                       
 * Purpose:  This functions enables the instrument driver user to clear the  
 *              error information the driver associates with the IVI session. Its 
 *              essentially an alias to the ClearError function.
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_ClearErrorInfo( ViSession vi )                                                        
{                                                                                                           
   return( Ivi_ClearErrorInfo( vi ) );
}

/**************************************************************************** 
 *  Function: AmetekDC_GetNextInterchangeWarning,
 *            AmetekDC_ResetInterchangeCheck, and
 *            AmetekDC_ClearInterchangeWarnings
 *  Purpose:  These functions allow the user to retrieve interchangeability
 *            warnings, reset the driver's interchangeability checking 
 *            state, and clear all previously logged interchangeability warnings.
 ****************************************************************************/
ViStatus _VI_FUNC AmetekDC_GetNextInterchangeWarning( ViSession vi, 
                                                    ViInt32 bufferSize, 
                                                    ViChar warnString[] )
{
   return( Ivi_GetNextInterchangeCheckString( vi, bufferSize, warnString ) );
}

ViStatus _VI_FUNC AmetekDC_ResetInterchangeCheck( ViSession vi )
{
   return( Ivi_ResetInterchangeCheck( vi ) );
}

ViStatus _VI_FUNC AmetekDC_ClearInterchangeWarnings( ViSession vi )
{
   return( Ivi_ClearInterchangeWarnings( vi ) );
}

/*****************************************************************************
 * Function: AmetekDC_InvalidateAllAttributes
 * Purpose:  This function invalidates the cached value of all attributes.
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_InvalidateAllAttributes( ViSession vi )
{
   return( Ivi_InvalidateAllAttributes( vi ) );
}

/*****************************************************************************
 * Function: AmetekDC_GetCommandTimeDelay                                  
 * Purpose:  This function queries the time delay used after each command that has
 *           been sent to the attached supply.  
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_GetCommandTimeDelay( ViSession vi, ViReal64* delayTimeSec )
{
   ViStatus error = VI_SUCCESS;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   checkErr( Ivi_GetAttributeViReal64( vi, VI_NULL,
                                       AMETEKDC_ATTR_CONFIGURABLE_DELAY,
                                       0, delayTimeSec ) );

Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 *  Function: AmetekDC_GetChannelName
 *  Purpose:  This function returns the highest-level channel name that
 *            corresponds to the specific driver channel string that is in
 *            the channel table at an index you specify.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_GetChannelName( ViSession vi, ViInt32 index,
                                         ViInt32 bufferSize,
                                         ViChar name[] )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  channelName;
   ViInt32        actualSize = 0;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
    
   if( bufferSize != 0 )
   {
      if(name == VI_NULL )
      {
         viCheckParm( AMETEKDC_ERROR_INVALID_PARAMETER, 4, 
                      "Null address for Channel Name" );
      }
   }  
   
   /* Get channel name */
   viCheckParm( Ivi_GetNthChannelString( vi, index, &channelName ),
                                         2, "Index" );
    
   actualSize = strlen( channelName ) + 1;
    
   /* Prepare return values */
   if( bufferSize == 0 )
   {
      error = actualSize;
   }
   else 
   {
      if( bufferSize > 0 )
      {
         if( actualSize > bufferSize )
         {
            error = actualSize;
            actualSize = bufferSize;
         }
      }
      
      memcpy(name, channelName, actualSize - 1);
      name[ actualSize - 1 ] = '\0';
   }

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_GetNextCoercionRecord                        
 * Purpose:  This function enables the instrument driver user to obtain
 *           the coercion information associated with the IVI session.                                                              
 *           This function retrieves and clears the oldest instance in which 
 *           the instrument driver coerced a value the instrument driver user
 *           specified to another value. 
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_GetNextCoercionRecord( ViSession vi,
                                                ViInt32 bufferSize,
                                                ViChar  record[] )
{
   return( Ivi_GetNextCoercionString( vi, bufferSize, record ) );
}

/*****************************************************************************
 * Function: AmetekDC_LockSession and AmetekDC_UnlockSession Functions                        
 * Purpose:  These functions enable the instrument driver user to lock the 
 *           session around a sequence of driver calls during which other
 *           execution threads must not disturb the instrument state.
 *                                                                          
 *           NOTE:  The callerHasLock parameter must be a local variable 
 *           initialized to VI_FALSE and passed by reference, or you can pass 
 *           VI_NULL.   
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_LockSession( ViSession vi, ViBoolean *callerHasLock )  
{                                              
   return( Ivi_LockSession( vi, callerHasLock ) );      
}                                              

ViStatus _VI_FUNC AmetekDC_UnlockSession( ViSession vi, ViBoolean *callerHasLock ) 
{                                              
   return( Ivi_UnlockSession( vi, callerHasLock ) );    
}  

/*****************************************************************************
 * Function: WriteInstrData and ReadInstrData Functions                      
 * Purpose:  These functions enable the instrument driver user to  
 *           write and read commands directly to and from the instrument.            
 * Note:     These functions bypass the IVI attribute state caching.  
 *           WriteInstrData invalidates the cached values for all 
 *           attributes.
 *
 * NOTE: Recommended function for IVI Instrument Driver.
 *
 *****************************************************************************/

ViStatus _VI_FUNC AmetekDC_WriteInstrData( ViSession vi, 
                                         ViConstString writeBuffer )   
{   
   return( Ivi_WriteInstrData( vi, writeBuffer ) );    
}   

ViStatus _VI_FUNC AmetekDC_ReadInstrData( ViSession vi, ViInt32 numBytes, 
                                        ViChar rdBuf[], ViInt32 *bytesRead )  
{   
   return( Ivi_ReadInstrData( vi, numBytes, rdBuf, bytesRead ) );   
} 



           
/*****************************************************************************
 *--------------- Configure Output Functions (Not Exported) -----------------*
 *****************************************************************************/
 
/*****************************************************************************
 * Function: AmetekDC_ConfigureOCP                                   
 * Purpose:  This function configures the over-current protection limit
 *           and sets the OCP state. You can set for a warning alarm only,
 *           or a warning alarm and output shutoff(trip) if an 
 *           over-current condition exists.
 *
 * Note:     To comply with the IVI standandard for the DCPwr Class, this 
 *           attribute is the same as the AMETEKDC_VAL_CURRENT_LIMIT_BEHAVIOR
 *           attribute. Therefore, only the IviDCPwr attribute was declared.
 *
 *           This function is called by AmetekDC_ConfigureProtectionSetpoint()
 *****************************************************************************/
static ViStatus AmetekDC_ConfigureOCP( ViSession vi, ViConstString channelName,
                                     ViBoolean behavior, ViReal64 limit )
{
   ViStatus error = VI_SUCCESS;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   // Due to the way the IviDCPwr class driver is implemented the behavior
   // change must appear before the change to the limit as changes to the
   // behavior can cause the current limit to change.
   viCheckParm( Ivi_SetAttributeViInt32( vi, channelName, 
                                         AMETEKDC_ATTR_CURRENT_LIMIT_BEHAVIOR, 
                                         0, behavior ), 4, "OCP Behavior" );
                                          
   viCheckParm( Ivi_SetAttributeViReal64( vi, channelName, 
                                          AMETEKDC_ATTR_OCP_LIMIT, 
                                          0, limit ), 5, "OCP Limit" );
    
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_ConfigureUVP                                   
 * Purpose:  This function configures the under-voltage protection limit
 *           and sets the UVP state. You can set for a warning alarm only,
 *           or a warning alarm and output shutoff(trip) if an 
 *           under-voltage condition exists.
 *
 *           This function is called by ConfigureProtectionSetpoint()
 *
 *****************************************************************************/
static ViStatus AmetekDC_ConfigureUVP( ViSession vi, ViConstString channelName,
                                     ViBoolean behavior, ViReal64 limit )
{
   ViStatus error = VI_SUCCESS;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );

   viCheckParm( Ivi_SetAttributeViBoolean( vi, channelName, 
                                           AMETEKDC_ATTR_UVP_BEHAVIOR, 
                                           0, behavior ), 4, "UVP Behavior" );
                                          
   viCheckParm( Ivi_SetAttributeViReal64( vi, channelName, 
                                          AMETEKDC_ATTR_UVP_LIMIT, 
                                          0, limit ), 5, "UVP Limit" );
    
   checkErr( AmetekDC_CheckStatus( vi, channelName ) );

Error:    
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 *-------------------- Utility Functions (Not Exported) ---------------------*
 *****************************************************************************/

/*****************************************************************************
 * Function: RemoveSurroundingWhiteSpace
 * Purpose:  This function removes surrounding white space around a string
 *
 *****************************************************************************/
static void RemoveSurroundingWhiteSpace( ViChar string[] )
{
   ViChar  *start = string,
           *end;
   ViInt32 lengthWithoutLeadingWS;

   if( start )
   {
      while( *start && isspace( ( unsigned char )*start ) )
      {
         start++;
      }

      lengthWithoutLeadingWS = strlen( start );
      memmove( string, start, lengthWithoutLeadingWS );
      end = string + lengthWithoutLeadingWS;
      
      while( end > string && isspace( ( unsigned char )*( end-1 ) ) )
      {
         end--;
      }
        
      *end = '\0';
   }
}

/*****************************************************************************
 *  Function: AmetekDC_GetDriverSetupOption
 *  Purpose:  Get the value of a DriverSetup option. The driver setup string
 *            is part of the option string the user passes to the
 *            AmetekDC_InitWithOptions function. Specifically it is the part
 *            that follows "DriverSetup=". The format of hte DriverSetup string
 *            is as follows <option name>: <option value> &< section option>:
 *            <second option value> etc....  
 *            Due to the way XG family hardware is setup there is a 
 *            requirement to use a master channel address in all communication.
 *            Thus, the required parameter is the currently selected MCHADDR. 
 *            Required Fields:
 *            1) "DriverSetup=MCHADDR:#"
 *
 *            Optional Fields (must be separated by '&'):
 *            1) "Model: Option Value"  (useful for testing in simulation mode)
 *            2) "Channels: #-#-#"      (available MCH use with RS-485 connected XG's)
 *            3) "BaudRate: 38400"
 *
 *  Example 1 "DriverSetup = MCHADDR:12&BaudRate:57600 &Model: XG 12-70" 
 *            (within existing quotes of the Option String)
 *
 *  Example 2 "DriverSetup = MCHADDR:8&Channels: 1-2-4-8-9"
 *            (within existing quotes of the Option String)
 *            
 *  Note:   The two optional parameter formats are mutually exclusive. 
 *            ie, you can either choose a model for simulation, or you can
 *            choose to have multiple channels. You cannot do both! If
 *            you choose to simulate with multiple channels, all the 
 *            instruments will default to be the value of the default model
 *             unit. If you do not specify a channel list, the instrument driver 
 *            creates a default list consisting of AMETEKDC_MASTER_CHANNEL.
 *            The channel AMETEKDC_MASTER_CHANNEL is always
 *            associated with the master unit and has a secondary address as
 *            specified by the MCHADDR option.
 *
 *
 *****************************************************************************/
static ViStatus AmetekDC_GetDriverSetupOption( ViSession vi, ViInt32 *modelType,
                                             ViChar channelString[] )
{
   ViStatus    error = VI_SUCCESS;
   ViBoolean   isValid = 0;
   ViChar      *channelPtr;
   ViChar      *tag,
                  setup[ BUFFER_SIZE ],
                  model[ BUFFER_SIZE ],
                  channels[ BUFFER_SIZE ],
                  strDataRate[ BUFFER_SIZE ],
 		     masterchannelAddress[ BUFFER_SIZE ];
   ViUInt16    interfaceType = VI_INTF_GPIB;
   ViUInt16    uiDataRate = AMETEKDC_IO_DEFAULT_DATARATE;
   	
   *modelType = AMETEKDC_MODEL_NOT_FOUND;
   strcpy( channelString, "No Channels" );
   
   checkErr( Ivi_GetAttributeViString( vi, VI_NULL, IVI_ATTR_DRIVER_SETUP, 
                                       0, BUFFER_SIZE, setup ) );
   
#ifdef __AMETEKDC_SPECIFIC_ADDRESS_HANDLING__	 
    /* Check for Model simulation tag */ 
    if( ( tag = strstr( setup, SETUP_MCHADDR_TAG ) ) != VI_NULL )
    {
   	    // Get the MCH_BUS_ADDRESS, the syntax of the configuration string is as
   	    // follows: "MCHADDR:<#>&<Additional driver options>".
   	    // This also allows the tag to be updated with the rest of the driver specific 
   	    // option allowing for the model option and the channel option to be specified.
          sscanf(tag, "%*[^:]:%[0-9]%[^\n]", masterchannelAddress, setup);
          RemoveSurroundingWhiteSpace( masterchannelAddress );
          RemoveSurroundingWhiteSpace( setup );		

	    // Verify the channel value is within range of the RS-485 bus addresses.
	    if( masterchannelAddress != VI_NULL) 
	    {
	 	    // Error invalid address.
	 	    checkErr( AmetekDC_VerifyChannelName( vi, masterchannelAddress, &isValid ) );
	    }

	    // Verify the correctness of the multichannel address, it must not 
	    // be the broadcast address (0).
	    if(( masterchannelAddress[0] == '0') || ( isValid == VI_FALSE) )
	    {
	 	    // Error, undefined multichannel address specified, throw a not 
	 	    // supported error to the IVI engine. 
	 	    viCheckErr( VI_ERROR_NSUP_ATTR_STATE );
	    }
    }
    else
    {
    	 // The user forgot the specify a mul
       sprintf(masterchannelAddress, "%u", AMETEKDC_DEFAULT_ADDRESS);
    }	
    // Store the multichannel address after the channel list has been established
    // at the end of the function.
#endif   //#ifdef __AMETEKDC_SPECIFIC_ADDRESS_HANDLING__	 


   /* Check for Initial Baud Rate Tag */ 
   if( ( tag = strstr( setup, SETUP_INTERFACE_BAUD ) ) != VI_NULL )
   {
      sscanf(tag, "%*[^:]:%[^&|\n]", strDataRate );      
      RemoveSurroundingWhiteSpace( strDataRate );

	uiDataRate = atoi(strDataRate);

	if( uiDataRate < 0)
      {
         ViChar buffer[ BUFFER_SIZE ];
         sprintf( buffer, "Bad value for Baudrate option in DriverSetup: %s", 
                  strDataRate );
                  
         viCheckErrElab( IVI_ERROR_BAD_OPTION_VALUE, buffer );
      }
   }
   
   /* Check for Model simulation tag */ 
   if( ( tag = strstr( setup, SETUP_MODEL_TAG ) ) != VI_NULL )
   {
      sscanf(tag, "%*[^:]:%[^&|\n]", model );      
      RemoveSurroundingWhiteSpace( model );

      if( Ivi_GetValueFromTable( AmetekDC_ModelTable, model, modelType ) 
          < VI_SUCCESS )
      {
         ViChar buffer[ BUFFER_SIZE ];
         sprintf( buffer, "Bad value for Model option in DriverSetup: %s", 
                  model );
                  
         viCheckErrElab( IVI_ERROR_BAD_OPTION_VALUE, buffer );
      }
   }
   
   /* Check for Channel List tag */
   if( ( tag = strstr( setup, SETUP_CHANNEL_TAG ) ) != VI_NULL )
   {
      sscanf(tag, "%*[^:]:%[^&|\n]", channels );
      RemoveSurroundingWhiteSpace( channels );
      
      /* Break the string into separate channels for verification and */ 
      /* reformatting */
      channelPtr = strtok( channels, "-" );
      
      if( channelPtr != VI_NULL )
      {  
         /* Check the channel for validity */
         checkErr( AmetekDC_VerifyChannelName( vi, channelPtr, &isValid ) );
      }
      
      if( isValid )
      {
         /* Create the first channel in the new channel string */
         strcpy( channelString, channelPtr );	  
      }
      else
      {
         viCheckParm( ( IVI_ERROR_INVALID_PARAMETER ), 3,
                        "Invalid channel entered in channel string, "
                        "or the channel string format is incorrect.\n\n"
                        "Valid channels range from 1 to 30" );
      }
      
      /* Continue parsing the string to obtain the next channel */
      channelPtr = strtok( VI_NULL, "-" );
      
      if( channelPtr != VI_NULL )
      {  
         /* Check the channel for validity */
         checkErr( AmetekDC_VerifyChannelName( vi, channelPtr, &isValid ) );
      }
      
      if( !isValid )
      {
         viCheckParm( ( IVI_ERROR_INVALID_PARAMETER ), 3,
                        "Invalid channel entered in channel string " 
                        "or the channel string format is incorrect.\n\n"
                        "Valid channels range from 1 to 30");
      }
      
      while( ( channelPtr != VI_NULL ) && ( isValid ) )
      {
         strcat( channelString, "," );
         strcat( channelString, channelPtr );
         channelPtr = strtok( VI_NULL, "-" );
         
         if( channelPtr != VI_NULL )
         {  
            checkErr( AmetekDC_VerifyChannelName( vi, channelPtr, &isValid ) );
         }
         
      }
      
      /* Check the isValid parameter to see if loop terminated due to a */
      /* bad string */
      if( !isValid )
      {
         viCheckParm( ( IVI_ERROR_INVALID_PARAMETER ), 3,
                        "Invalid channel entered in channel string"
                        "or the channel string format is incorrect.\n\n"
                        "Valid channels range from 1 to 30");
      }

   }/* end else if( ( tag = strstr( setup, SETUP_CHANNEL_TAG ) ) != VI_NULL ) */
   else; /* Do nothing */

   // Need to add the master channel value.
   if( strcmp( channelString, "No Channels" ) == 0 )
   {
      /* There is not a user specified channel list for RS-485, so create */
      /* default channel of AMETEKDC_MASTER_CHANNEL for communication to the Master device */
      strcpy( channelString, AMETEKDC_MASTER_CHANNEL );
   }
   else
   {
      /* There is a user specified channel list for RS-485, so append AMETEKDC_MASTER_CHANNEL */
      /* for communication to the Master device (GPIB or RS232 unit) */
      strcat( channelString, "," );
      strcat( channelString, AMETEKDC_MASTER_CHANNEL);  
   }

   // Build the channel table now that all channels have been enumerated.
   checkErr( Ivi_BuildChannelTable( vi, channelString, VI_FALSE, VI_NULL ) );

#ifdef __AMETEKDC_SPECIFIC_ADDRESS_HANDLING__	 
   // Need to add this attribute here so that the driver can store  the address.  
   // This centralizes the driver configuration handling even though it breaks the 
   // model of adding all driver specific attributes in AmetekDC_InitAttributes().
   checkErr (Ivi_AddAttributeViString (vi, AMETEKDC_ATTR_BUS_ADDRESS,
	                                    "AMETEKDC_ATTR_BUS_ADDRESS", "2",
	                                    IVI_VAL_ALWAYS_CACHE, VI_NULL, VI_NULL));

   // At the present moment the first channel will be the master channel.
   AmetekDC_SetAttributeViString(vi, VI_NULL, AMETEKDC_ATTR_BUS_ADDRESS, masterchannelAddress );


   checkErr (Ivi_AddAttributeViInt32 (vi, AMETEKDC_ATTR_DATARATE,
	                                   "AMETEKDC_ATTR_DATARATE", 9600,
	                                   IVI_VAL_ALWAYS_CACHE | IVI_VAL_USE_CALLBACKS_FOR_SIMULATION,
	                                   VI_NULL, VI_NULL, &attrDataRateRangeTable));   

   // At the present moment the first channel will be the master channel.
   AmetekDC_SetAttributeViInt32(vi, VI_NULL, AMETEKDC_ATTR_DATARATE, uiDataRate);

   
#endif   //#ifdef __AMETEKDC_SPECIFIC_ADDRESS_HANDLING__	 
   
Error:
   return( error );
}

/*****************************************************************************
 *  Function: AmetekDC_VerifyChannelName
 *  Purpose:  Takes the parsed channel string from AmetekDC_GetDriverSetupOption
 *            and verifies it to be between 1 and 30.
 *
 *****************************************************************************/ 
static ViStatus AmetekDC_VerifyChannelName( ViSession vi, 
                                          ViConstString channelName,
                                          ViBoolean *isValid )
{
   ViStatus error = VI_SUCCESS;
   ViInt32  channelInteger = 0;
   
   channelInteger = atoi( channelName );
   
   if( ( channelInteger > 0 ) && ( channelInteger <= 31 ) ) 

   {    
      *isValid = VI_TRUE;
   }
   else
   {
      *isValid = VI_FALSE;
   }
   
Error:
   return( error );
}
             
/**************************************************************************
 *  Function: AmetekDCQueryID_CallbackHelper
 *  Purpose:  Query the instrument for its id string.
 *
 *****************************************************************************/
static ViStatus AmetekDCQueryID_CallbackHelper( ViSession vi, ViSession io,
                                ViConstString channelName,
                                ViInt32 bufferSize,
                                ViChar buffer[] )
{
   ViStatus  error = VI_SUCCESS;
   ViUInt32  readCount;

   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SYST_IDEN, "?") );
   
	// Expect a timeout here as I don't want to put an expected number of characters.
   	viCheckErr(viRead( io, buffer, bufferSize - 1, &readCount ));
   	buffer[ readCount ] = '\0';   
   
Error:   
   Ivi_UnlockSession(vi, VI_NULL);
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_GetModelFromBuffer
 * Purpose:  This function returns model number from buffer. If table 
 *           AmetekDC_ModelTable does not contains this model, function
 *           returns VI_ERROR_FAIL_ID_QUERY;
 *
 *           Note:  Call this function only when the session is locked.
 *****************************************************************************/
static ViStatus AmetekDC_GetModelFromBuffer( ViSession vi, ViConstString buffer,
                                           ViInt32 *modelType )
{
   ViStatus error = VI_SUCCESS;
   ViChar   tmpBuffer[ SMALL_BUFF_SIZE ];
 
   sscanf( buffer,"%*[^,], %[^,]", tmpBuffer );
   
   if( Ivi_GetValueFromTable( AmetekDC_ModelTable, tmpBuffer, modelType ) 
       < VI_SUCCESS )
   {
      viCheckErr( VI_ERROR_FAIL_ID_QUERY );
   }
Error:
   return( error );
}

 /*****************************************************************************
 * Function: AmetekDC_CheckStatus                       
 * Purpose:  This function checks the status of the instrument to detect 
 *           whether the instrument has encountered an error.  This function  
 *           is called at the end of most exported driver functions.  If the    
 *           instrument reports an error, this function returns      
 *           IVI_ERROR_INSTR_SPECIFIC.  The user can set the 
 *           IVI_ATTR_QUERY_INSTR_STATUS attribute to VI_FALSE to disable this 
 *           check and increase execution speed.                                   
 *
 *           Note:  Call this function only when the session is locked.
 *
 *****************************************************************************/
static ViStatus AmetekDC_CheckStatus( ViSession vi, ViConstString channelName )
{
   ViStatus    error = VI_SUCCESS;

   if( Ivi_QueryInstrStatus( vi ) && Ivi_NeedToCheckStatus( vi ) 
       && !Ivi_Simulating( vi ) )
   {
       Delay(AMETEKDC_MIN_CMD_DELAY_SEC+0.005);//0.025 sec delay is added to avoid accasional error.
	   checkErr( AmetekDC_CheckStatusCallback( vi, Ivi_IOSession( vi ), 
                                            channelName ) );
       checkErr( Ivi_SetNeedToCheckStatus( vi, VI_FALSE ) );
   }
        
Error:
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_Delay                                                
 * Purpose:  This function delay's processing by a time amount set by the user.
 *           The default delay is 0.015 seconds. The user can set this delay time
 *           when they are using a fast computer that sends commands to the
 *           XG faster than it can process them.
 *
 *           This function is only meant to be used after a viPrintf() has
 *           been executed, and the SCPI command sent did not include a channel 
 *           designator (non-CANbus I/O).
 *
 * Note:     Call this function only when the session is locked.
 *****************************************************************************/
static ViStatus AmetekDC_Delay( ViSession vi )
{
   ViStatus    error = VI_SUCCESS;
   ViReal64    userDelay = 0.0;

   checkErr( Ivi_GetAttributeViReal64( vi, VI_NULL, 
                                       AMETEKDC_ATTR_CONFIGURABLE_DELAY,
                                       0, &userDelay ) );
   Delay( userDelay );                                       
        
Error:
   return( error );
}


/*****************************************************************************
 *  Function: AmetekDC_SetCommBaudRate
 *  Purpose:  Sets the data rate that will be used to communicate to the XG.
 *
 *****************************************************************************/ 
ViStatus _VI_FUNC AmetekDC_SetCommBaudRate( ViSession vi, ViInt32 uiDataRate )
{
   ViStatus error = VI_SUCCESS;

   // At the present moment the first channel will be the master channel.
   checkErr(AmetekDC_SetAttributeViInt32(vi, VI_NULL, AMETEKDC_ATTR_DATARATE, uiDataRate));

   if( !Ivi_Simulating( vi ) )
   {
      ViSession   io = VI_NULL;        
	  
      /* io session owned by driver now */
      checkErr( Ivi_LockSession( vi, VI_NULL ) );

      // Get the io session to update the baudrate.
     checkErr( Ivi_GetAttributeViSession( vi, VI_NULL, IVI_ATTR_IO_SESSION, 0, &io ) );
   
	// Configure the data rate of the serial connection.
	viCheckErr( viSetAttribute( io, VI_ATTR_ASRL_BAUD, uiDataRate));
   }
   
Error:
 	Ivi_UnlockSession( vi, VI_NULL );
   return( error );   
}


/*****************************************************************************
 *  Function: AmetekDC_GetCommBaudRate
 *  Purpose:  Gets the currently configured data rate that is used to communicate 
 *               to the XG. 
 *
 *****************************************************************************/ 
ViStatus _VI_FUNC AmetekDC_GetCommBaudRate( ViSession vi, ViInt32 *uiDataRate )
{
   ViStatus error = VI_SUCCESS;
   
   // At the present moment the first channel will be the master channel.
   checkErr( AmetekDC_GetAttributeViInt32(vi, VI_NULL, AMETEKDC_ATTR_DATARATE, uiDataRate));


Error:
   return( error );   
}

/*****************************************************************************
 * Function: AmetekDC_DefaultInstrSetup                                               
 * Purpose:  This function sends a default setup to the instrument.  The    
 *           AmetekDC_reset function calls this function.  The 
 *           AmetekDC_IviInit function calls this function when the
 *           user passes VI_FALSE for the reset parameter.  This function is 
 *           useful for configuring settings that other instrument driver 
 *           functions require.    
 * Note:     Call this function only when the session is locked.
 *****************************************************************************/
static ViStatus AmetekDC_DefaultInstrSetup( ViSession vi )
{
   ViStatus    error = VI_SUCCESS;
   ViReal64    max=0;
   
   /* Invalidate all attributes */
   checkErr( Ivi_InvalidateAllAttributes( vi ) );
    
Error:
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_SendBooleanCmd                       
 * Purpose:  This function builds the parameter for command to be sent,
 *              then calls the SendCmd function which builds and send the 
 *              command on to its destination.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SendBooleanCmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
                                             ViUInt32 cmdIdx,
                                             ViBoolean value)
{
   ViStatus error = VI_SUCCESS;
   ViChar    ascParamter[] = " OFF";

   // Convert the value into the output format (string).
   if( value != 0)
   {
   	ascParamter[2] = 'N'; 
   	ascParamter[3] = 0; 
   }

   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, cmdIdx, ascParamter) );
   

Error:
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_SendInt16Cmd                       
 * Purpose:  This function builds the parameter for command to be sent,
 *              then calls the SendCmd function which builds and send the 
 *              command on to its destination.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SendInt16Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
                                             ViUInt32 cmdIdx,
                                             ViInt16 value)
{
   ViStatus error = VI_SUCCESS;
   ViChar ascParamter[ SMALL_BUFF_SIZE ];

   // Format real number parameter into a string.
   viSPrintf(io, ascParamter, AMETEKDC_SHORT_PARAM_OUTP_FORMAT, value);
   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, cmdIdx, ascParamter) );

   
Error:
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_SendUInt16Cmd                       
 * Purpose:  This function builds the parameter for command to be sent,
 *              then calls the SendCmd function which builds and send the 
 *              command on to its destination.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SendUInt16Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
                                             ViUInt32 cmdIdx,
                                             ViUInt16 value)
{
   ViStatus error = VI_SUCCESS;
   ViChar ascParamter[ SMALL_BUFF_SIZE ];

   // Format real number parameter into a string.
   viSPrintf(io, ascParamter, AMETEKDC_USHORT_PARAM_OUTP_FORMAT, value);
   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, cmdIdx, ascParamter) );

   
Error:
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_SendInt32Cmd                       
 * Purpose:  This function builds the parameter for command to be sent,
 *              then calls the SendCmd function which builds and send the 
 *              command on to its destination.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SendInt32Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
                                             ViUInt32 cmdIdx,
                                             ViInt32 value)
{
   ViStatus error = VI_SUCCESS;
   ViChar ascParamter[ SMALL_BUFF_SIZE ];

   // Format real number parameter into a string.
   viSPrintf(io, ascParamter, AMETEKDC_LONG_PARAM_OUTP_FORMAT, value);
   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, cmdIdx, ascParamter) );
                                             
Error:
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_SendUInt32Cmd                       
 * Purpose:  This function builds the parameter for command to be sent,
 *              then calls the SendCmd function which builds and send the 
 *              command on to its destination.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SendUInt32Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
                                             ViUInt32 cmdIdx,
                                             ViUInt32 value)
{
   ViStatus error = VI_SUCCESS;
   ViChar ascParamter[ SMALL_BUFF_SIZE ];

   // Format real number parameter into a string.
   viSPrintf(io, ascParamter, AMETEKDC_ULONG_PARAM_OUTP_FORMAT, value);
   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, cmdIdx, ascParamter) );
                                             
Error:
   return( error );
}
/*****************************************************************************
 * Function: AmetekDC_SendReal64Cmd                       
 * Purpose:  This function builds the parameter for command to be sent,
 *              then calls the SendCmd function which builds and send the 
 *              command on to its destination.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SendReal64Cmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
                                             ViUInt32 cmdIdx,
                                             ViReal64 value)
{
   ViStatus error = VI_SUCCESS;
   ViChar ascParamter[ SMALL_BUFF_SIZE ];

   // Format real number parameter into a string.
   viSPrintf(io, ascParamter, AMETEKDC_REAL_PARAM_OUTP_FORMAT, value);
   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, cmdIdx, ascParamter) );
   
Error:
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_SendStringCmd                       
 * Purpose:  This function builds the parameter for command to be sent,
 *              then calls the SendCmd function which builds and send the 
 *              command on to its destination.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SendStringCmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
                                             ViUInt32 cmdIdx,
                                             ViConstString cmdParameters)
{
   ViStatus error = VI_SUCCESS;
   ViChar ascParameter[ SMALL_BUFF_SIZE ];

   // Format real number parameter into a string.
   viSPrintf(io, ascParameter, AMETEKDC_STRING_PARAM_OUTP_FORMAT, cmdParameters);
   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, cmdIdx, ascParameter) );

Error:
   return( error );
}

                     

/*****************************************************************************
 * Function: AmetekDC_SendCmd                       
 * Purpose:  This function builds the command to be sent from the command, 
 *              the destination channel and the parameters to send. This 
 *              function should be used in all cases for this  driver so that 
 *              correct addressing can be handled. See notes on XG addressing.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_SendCmd( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName,
                                             ViUInt32 cmdIdx,
                                             ViConstString cmdParameters )
{
   ViStatus error = VI_SUCCESS;
   ViString partialCmdBuffer;
   ViChar cmdBuffer[ BUFFER_SIZE ];
   ViChar cmdMsg[ BUFFER_SIZE ];
   ViChar channelAddr [ SMALL_BUFF_SIZE ];
   ViPJobId pJobID = 0;

  checkErr( Ivi_LockSession( vi, VI_NULL ) ); 

   // Check if we are simulating.  If so return success.
   if( Ivi_Simulating( vi ) )
   {   
	return( error );
   }

     // Get the comamnd from the command string range table.
   viCheckParm( Ivi_GetViInt32EntryFromValue( cmdIdx, 
                                              &scpiCmdStringRangeTable,
                                              VI_NULL, VI_NULL, VI_NULL, 
                                              VI_NULL, &partialCmdBuffer, VI_NULL ), 
                                              1, "Command Index" );
                                              
   // Get the string of the command to be sent and add parameters.
   viSPrintf(io, cmdMsg, "%s%s", partialCmdBuffer, cmdParameters);

   // Add the manditory multichannel address.
   if( ( strcmp( channelName, AMETEKDC_MASTER_CHANNEL ) == 0 ) || 
       ( strcmp( channelName, "a" ) == 0 ) )
   {
      // Using the master address.  Look up the bus address associate with the master address.
      AmetekDC_GetAttributeViString(vi, VI_NULL, 
                                                 AMETEKDC_ATTR_BUS_ADDRESS,
                                                 SMALL_BUFF_SIZE,
                                                 channelAddr );
	  						   
      viCheckErr( viSPrintf( io, cmdBuffer, cmdMsg, channelAddr) );
   }
   else
   {
      viCheckErr( viSPrintf( io, cmdBuffer, cmdMsg, channelName ) );
   }

   /* add <CR> to end of command for ENET_IF */
   if(user_if == ENET_IF)
		strcat(cmdBuffer, "\r");
	
   //Send the command to the XG.
   viCheckErr(viWriteAsync(io,cmdBuffer, strlen(cmdBuffer), pJobID));
      
   // Make sure the message has been completely sent before proceding.
   viCheckErr(viWaitOnEvent(io, VI_EVENT_IO_COMPLETION, 
   	                              AMETEKDC_IO_COMPLETION_TIMEOUT_MS,
   	                              VI_NULL, VI_NULL));

   // Delay the time necessary for the main loop to recieve and begin handling 
   // the command.
   Delay(AMETEKDC_MAINLOOP_LATENCY_SEC);
  
   // Wait for required delay before sending the next command.
   checkErr( AmetekDC_Delay( vi ) ); /* User-Configurable delay */      

Error:
   Ivi_UnlockSession( vi, VI_NULL ); 
   return( error );
}

/*****************************************************************************
 * Function: AmetekDC_SetRangeTableAddress                       
 * Purpose:  This function determines which range tables each channel based
 *           instrument will be linked to. The range tables that are determined
 *           by the model are the voltage and current ranges as well as the OVP
 *           range.
 *
 *****************************************************************************/
static ViStatus AmetekDC_SetRangeTableAddress( ViSession vi, 
                                             ViConstString channelString )
{
   ViStatus error = VI_SUCCESS;
   ViChar   channelList[ BUFFER_SIZE ];
   ViChar   *channel;
   ViInt32  modelType;
   
   checkErr( Ivi_LockSession( vi, VI_NULL ) );
   
   strcpy( channelList, channelString );
   channel = strtok( channelList, "," );
   
   do
   {  
      checkErr( Ivi_GetAttributeViInt32( vi, channel, AMETEKDC_ATTR_MODEL_TYPE, 0,
                                         &modelType ) );

	// Do range check to make sure you don't index off the end of an array.
      if(modelType < XTR_NUMBER_MODELS)//XG_NUMBER_MODELS)
      	{
  		// Using lookup table for setting these parameters.
  		  checkErr( Ivi_SetAttributeViAddr( vi, channel,
                                    AMETEKDC_ATTR_VOLTAGE_RANGE_TABLE_ADDRESS, 0,
                                    AmetekDC_kazModelOutputRangeTable[modelType].addrVoltRangeTbl ) );
                                 
	      checkErr( Ivi_SetAttributeViAddr( vi, channel,
                                    AMETEKDC_ATTR_CURRENT_RANGE_TABLE_ADDRESS, 0,
                                    AmetekDC_kazModelOutputRangeTable[modelType].addrCurrRangeTbl ) );
                                    
	      checkErr( Ivi_SetAttributeViAddr( vi, channel,
                                    AMETEKDC_ATTR_OVP_RANGE_TABLE_ADDRESS, 0,
                                    AmetekDC_kazModelOutputRangeTable[modelType].addrOvpRangeTbl ) );                                    
      	}
	else
	{
		// Index is out of range, an error should be thrown. Then the default 
		
	 	// Using lookup table for setting these parameters.
      		checkErr( Ivi_SetAttributeViAddr( vi, channel,
                                    AMETEKDC_ATTR_VOLTAGE_RANGE_TABLE_ADDRESS, 0,
                                    AmetekDC_kazModelOutputRangeTable[AMETEKDC_DEFAULT_MODEL_TYPE].addrVoltRangeTbl ) );
                                 
	      checkErr( Ivi_SetAttributeViAddr( vi, channel,
                                    AMETEKDC_ATTR_CURRENT_RANGE_TABLE_ADDRESS, 0,
                                    AmetekDC_kazModelOutputRangeTable[AMETEKDC_DEFAULT_MODEL_TYPE].addrCurrRangeTbl ) );	
                                    
	      checkErr( Ivi_SetAttributeViAddr( vi, channel,
                                    AMETEKDC_ATTR_OVP_RANGE_TABLE_ADDRESS, 0,
                                    AmetekDC_kazModelOutputRangeTable[modelType].addrOvpRangeTbl ) );                                    
	}
	    
      channel = strtok( VI_NULL, "," );
   } 
   while( channel != VI_NULL );
   
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


/*****************************************************************************
 * Function: AmetekDC_GetError Function                       
 * Purpose:  This functions enables the instrument driver user to get the  
 *              error information the driver associates with the IVI session.
 *
 * NOTE: Required function for IVI Instrument Driver.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_GetError( ViSession vi, 
                                   ViStatus *errorCode, 
                                   ViInt32 bufferSize,
                                   ViChar description[] )
{
   ViStatus    error = VI_SUCCESS;
   ViStatus    primary = VI_SUCCESS,
               secondary = VI_SUCCESS;
   ViChar      elaboration[ 256 ] = "";
   ViChar      errorMessage[ 1024 ] = "";
   ViChar      *appendPoint = errorMessage;
   ViInt32     actualSize = 0;
   ViBoolean   locked = VI_FALSE;

   /* Lock Session */      
   if( vi != 0 )
   {
      checkErr( Ivi_LockSession( vi, &locked ) );
   }

   /* Test for nulls and acquire error data */
   if( bufferSize != 0 )
   {
      if( errorCode == VI_NULL )
      {
         viCheckParm( AMETEKDC_ERROR_INVALID_PARAMETER, 3, 
                      "Null address for Error" );
      }
      
      if( description == VI_NULL )
      {
         viCheckParm( AMETEKDC_ERROR_INVALID_PARAMETER, 4, 
                      "Null address for Description" );
      }
      
      checkErr( Ivi_GetErrorInfo( vi, &primary, &secondary, elaboration ) );
   }
   else
   {
      checkErr( Ivi_GetAttributeViString( vi, VI_NULL, 
                                          AMETEKDC_ATTR_ERROR_ELABORATION, 
                                          0, 256, elaboration ) );
                                          
      checkErr( Ivi_GetAttributeViInt32( vi, VI_NULL, 
                                         AMETEKDC_ATTR_SECONDARY_ERROR, 
                                         0, &secondary ) );
                                         
      checkErr( Ivi_GetAttributeViInt32( vi, VI_NULL, 
                                         AMETEKDC_ATTR_PRIMARY_ERROR, 
                                         0, &primary ) );
   }
      
   /* Format data */
   if( primary != VI_SUCCESS )
   {
      ViChar msg[ 256 ] = "";
      checkErr( AmetekDC_error_message( vi, primary, msg ) );
      
      appendPoint += sprintf( appendPoint, "Primary Error:(Hex 0x%08X) %s", 
                              primary, msg);
   }
   
   if( secondary != VI_SUCCESS )
   {
      ViChar msg[ 256 ] = "";
      checkErr( AmetekDC_error_message( vi, secondary, msg ) );
      
      appendPoint += sprintf( appendPoint, "\nSecondary Error:(Hex 0x%08X) %s", 
                              secondary, msg );
   }
   
   if( elaboration[ 0 ] )
   {
      sprintf( appendPoint, "\nElaboration: %s", elaboration );
   }
   
   actualSize = strlen( errorMessage ) + 1;
   
   /* Prepare return values */
   if( bufferSize == 0 )
   {
      error = actualSize;
   }
   else 
   {
      if( bufferSize > 0 )
      {
         if( actualSize > bufferSize )
         {
            error = actualSize;
            actualSize = bufferSize;
         }
      }
      
      memcpy( description, errorMessage, actualSize - 1 );
      description[ actualSize - 1 ] = '\0';
   }
   
   if( errorCode ) 
   {
      *errorCode = primary;
   }
   
Error:
   /* Unlock Session */
   Ivi_UnlockSession( vi, &locked );
   return( error );
} 


/*****************************************************************************
 * Function: AmetekDC_GetErrorInfo and AmetekDC_ClearErrorInfo Functions                       
 * Purpose:  These functions enable the instrument driver user to  
 *           get or clear the error information the driver associates with the
 *           IVI session.
 *
 * Note:     These functions have been Deprecated from the IVI Specifications.
 *
 *****************************************************************************/
ViStatus _VI_FUNC AmetekDC_GetErrorInfo( ViSession vi, 
                                       ViStatus *primaryError, 
                                       ViStatus *secondaryError, 
                                       ViChar errorElaboration[ 256 ] )  
{                                                                                                           
   return( Ivi_GetErrorInfo( vi, primaryError, secondaryError, 
                             errorElaboration) );
}                                                                                                           



/*****************************************************************************
 *------------------------ Global Session Callbacks -------------------------*
 *****************************************************************************/

 /*****************************************************************************
 * Function: AmetekDC_CheckStatusCallback                       
 * Purpose:  This function queries the instrument to determine if it has 
 *           encountered an error.  If the instrument has encountered an 
 *           error, this function returns the IVI_ERROR_INSTRUMENT_SPECIFIC 
 *           error code.  This function is called by the 
 *           AmetekDC_CheckStatus utility function.  The 
 *           Ivi_SetAttribute and Ivi_GetAttribute functions invoke this 
 *           function when the optionFlags parameter includes the
 *           IVI_VAL_DIRECT_USER_CALL flag.
 *           
 *           The user can disable calls to this function by setting the 
 *           IVI_ATTR_QUERY_INSTR_STATUS attribute to VI_FALSE.  The driver can 
 *           disable the check status callback for a particular attribute by 
 *           setting the IVI_VAL_DONT_CHECK_STATUS flag.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDC_CheckStatusCallback( ViSession vi, ViSession io,
                                                     ViConstString channelName )
{
   ViStatus       error = VI_SUCCESS;
   ViConstString  coercedChannel;
   ViInt32        uiSESR = 0; 
   
   checkErr( Ivi_CoerceChannelName( vi, channelName, &coercedChannel ) );

   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, coercedChannel, AMETEKDC_VAL_STAT_STANDARD_EVENT, "?") );
   viCheckErr( viScanf( io, AMETEKDC_ULONG_PARAM_INP_FORMAT, &uiSESR ) ); 
       

   if( IEEE_488_2_ERROR_BITS & uiSESR )
   {
      viCheckErr( IVI_ERROR_INSTR_SPECIFIC );     
   }
            
Error:
   return( error );
}


/*****************************************************************************
 *----------------- Attribute Range Tables and Callbacks --------------------*
 *****************************************************************************/

/*- AMETEKDC_ATTR_ID_QUERY_RESPONSE -*/
static ViStatus _VI_FUNC AmetekDCAttrIdQueryResponse_ReadCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             const ViConstString cacheValue )
{
   ViStatus    error = VI_SUCCESS;
   ViChar      rdBuffer[ BUFFER_SIZE ];

   checkErr( AmetekDCQueryID_CallbackHelper( vi, io, channelName, BUFFER_SIZE, rdBuffer ) );
   checkErr( Ivi_SetValInStringCallback( vi, attributeId, rdBuffer ) );
 
Error:
   return( error );
}

/*- AMETEKDC_ATTR_SPECIFIC_DRIVER_REVISION -*/
static ViStatus _VI_FUNC AmetekDCAttrDriverRevision_ReadCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             const ViConstString cacheValue )
{
   ViStatus    error = VI_SUCCESS;
   ViChar      driverRev[ 256 ];
    
   sprintf( driverRev, 
            "Driver: AmetekDC %.2f, Compiler: %s %3.2f, "
            "Components: IVIEngine %.2f, VISA-Spec %.2f",
            (AMETEKDC_MAJOR_VERSION + AMETEKDC_MINOR_VERSION/1000.0), 
            IVI_COMPILER_NAME, IVI_COMPILER_VER_NUM, 
            IVI_ENGINE_MAJOR_VERSION + IVI_ENGINE_MINOR_VERSION/1000.0, 
            Ivi_ConvertVISAVer( VI_SPEC_VERSION ) );

   checkErr( Ivi_SetValInStringCallback( vi, attributeId, driverRev ) );
   
Error:
   return( error );
}

/*- AMETEKDC_ATTR_INSTRUMENT_FIRMWARE_REVISION -*/
static ViStatus _VI_FUNC AmetekDCAttrInstrumentFirmwareRevision_ReadCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             const ViConstString cacheValue )
{
   ViStatus    error = VI_SUCCESS;
   ViChar      rdBuffer[ SMALL_BUFF_SIZE ];
   ViChar      idQ[ BUFFER_SIZE ];
    
   checkErr( Ivi_GetAttributeViString( vi, AMETEKDC_MASTER_CHANNEL, AMETEKDC_ATTR_ID_QUERY_RESPONSE,
                                       0, BUFFER_SIZE, idQ ) );
                                       
   sscanf( idQ,"%*[^,],%*[^,],%*[^,],%[^\n]", rdBuffer ); 
    
   checkErr( Ivi_SetValInStringCallback( vi, attributeId, rdBuffer ) );
    
Error:
   return( error );
}

/*- AMETEKDC_ATTR_INSTRUMENT_MANUFACTURER -*/
static ViStatus _VI_FUNC AmetekDCAttrInstrumentManufacturer_ReadCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             const ViConstString cacheValue )
{
   ViStatus    error = VI_SUCCESS;
   ViChar      rdBuffer[ BUFFER_SIZE ];
   ViChar      idQ[ BUFFER_SIZE ];
    
   checkErr( Ivi_GetAttributeViString( vi, AMETEKDC_MASTER_CHANNEL, AMETEKDC_ATTR_ID_QUERY_RESPONSE,
                                       0, BUFFER_SIZE, idQ ) );
                                       
   sscanf( idQ, "%256[^,]", rdBuffer );

   checkErr( Ivi_SetValInStringCallback( vi, attributeId, rdBuffer ) );
    
Error:
   return( error );
}

/*- AMETEKDC_ATTR_INSTRUMENT_MODEL -*/
static ViStatus _VI_FUNC AmetekDCAttrInstrumentModel_ReadCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             const ViConstString cacheValue )
{
   ViStatus    error = VI_SUCCESS;
   ViChar      rdBuffer[ BUFFER_SIZE ];
   ViChar      idQ[ BUFFER_SIZE ];
    
   checkErr( Ivi_GetAttributeViString( vi, AMETEKDC_MASTER_CHANNEL, AMETEKDC_ATTR_ID_QUERY_RESPONSE,
                                       0, BUFFER_SIZE, idQ ) );
                                        
   sscanf( idQ, "%*[^,],%256[^,]", rdBuffer );

   checkErr( Ivi_SetValInStringCallback( vi, attributeId, rdBuffer ) );
    
Error:
   return( error );
}

/*- AMETEKDC_ATTR_MULTICHANNEL_FIRMWARE_REVISION -*/
static ViStatus _VI_FUNC AmetekDCAttrMultichannelFirmwareRevision_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             const ViConstString cacheValue )
{
   ViStatus    error = VI_SUCCESS;
   ViChar      rdBuffer[ SMALL_BUFF_SIZE ];
   ViChar      idQ[ BUFFER_SIZE ];
    
   checkErr( Ivi_GetAttributeViString( vi, channelName, AMETEKDC_ATTR_ID_QUERY_RESPONSE,
                                       0, BUFFER_SIZE, idQ ) );
                                       
   sscanf( idQ,"%*[^,],%*[^,],%*[^,],%[^\n]", rdBuffer ); 
    
   checkErr( Ivi_SetValInStringCallback( vi, attributeId, rdBuffer ) );
   
Error:
   return( error );
}

/*- AMETEKDC_ATTR_MULTICHANNEL_MODEL -*/
static ViStatus _VI_FUNC AmetekDCAttrMultichannelModel_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             const ViConstString cacheValue )
{
   ViStatus    error = VI_SUCCESS;
   ViChar      rdBuffer[ BUFFER_SIZE ];
   ViChar      idQ[ BUFFER_SIZE ];
    
   checkErr( Ivi_GetAttributeViString( vi, channelName, AMETEKDC_ATTR_ID_QUERY_RESPONSE,
                                       0, BUFFER_SIZE, idQ ) );
                                        
   sscanf( idQ, "%*[^,],%256[^,]", rdBuffer );

   checkErr( Ivi_SetValInStringCallback( vi, attributeId, rdBuffer ) );

Error:
   return( error );
}

/*- AMETEKDC_ATTR_OUTPUT_ENABLED -*/
/**************************************************************************
 *  Function: AmetekDCAttrOutputEnabled_WriteCallback
 *  Purpose:  Set the main output of the power supply on/off.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOutputEnabled_WriteCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViBoolean value )
{
   ViStatus error = VI_SUCCESS;
   
   // Send the command.
   viCheckErr( AmetekDC_SendBooleanCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_STATE, value) );

Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrOutputEnabled_ReadCallback
 *  Purpose:  Query the state of main output of the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOutputEnabled_ReadCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViBoolean* value )
{
   ViStatus error = VI_SUCCESS;

   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_STATE, "?") );
   viCheckErr( viScanf( io, AMETEKDC_BOOL_PARAM_INP_FORMAT, value ) );  
   

Error:
   return( error );
}

/*- AMETEKDC_ATTR_VOLTAGE_LEVEL -*/
/**************************************************************************
 *  Function: AmetekDCAttrVoltageLevel_WriteCallback
 *  Purpose:  Set the voltage set point of the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrVoltageLevel_WriteCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViReal64 value )
{
   ViStatus error = VI_SUCCESS;

   // Send the command.
   checkErr( AmetekDC_SendReal64Cmd(vi, io, channelName, AMETEKDC_VAL_SOUR_VOLT, value) );
   
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrVoltageLevel_ReadCallback
 *  Purpose:  Query the voltage set point of the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrVoltageLevel_ReadCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViReal64* value )
{
   ViStatus error = VI_SUCCESS;
   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SOUR_VOLT, "?") );
   viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_INP_FORMAT, value ) ); 

	
Error:
   return( error );
}

/*- AMETEKDC_ATTR_CURRENT_LIMIT-*/
/**************************************************************************
 *  Function: AmetekDCAttrCurrentLimit_WriteCallback
 *  Purpose:  Set the current set point of the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrCurrentLimit_WriteCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViReal64 value )
{
   ViStatus error = VI_SUCCESS;
   
   // Send the command.
   checkErr( AmetekDC_SendReal64Cmd(vi, io, channelName, AMETEKDC_VAL_SOUR_CURR, value) );   

Error: 
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrCurrentLimit_ReadCallback
 *  Purpose:  Query the current set point of the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrCurrentLimit_ReadCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViReal64* value )
{
   ViStatus error = VI_SUCCESS;

   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SOUR_CURR, "?") );
   viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_INP_FORMAT, value ) );

Error:
   return( error );
}


/***************************************************************************
 * ------------------ Triggered Limits Callback's -------------------------*
 ***************************************************************************/
/*- AMETEKDC_ATTR_TRIGGERED_VOLTAGE_LEVEL -*/

   
/*- AMETEKDC_ATTR_TRIGGERED_CURRENT_LIMIT -*/




/*- AMETEKDC_ATTR_TRIGGER_SOURCE -*/


/*- AMETEKDC_ATTR_FOLD_DELAY -*/
static IviRangeTableEntry attrFoldDelayRangeTableEntries[] =
	{
		{0.5, 50.0, 0, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable attrFoldDelayRangeTable =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        attrFoldDelayRangeTableEntries,
	};

/**************************************************************************
 *  Function: AmetekDCAttrFoldDelay_ReadCallback
 *  Purpose:  Query the foldback delay of the power supply.
 *
 * Version: 1.00            Date: Aug 10,2006		      Author: Chris Singer
 *    - Initial revision
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrFoldDelay_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 *value )
{
   ViStatus error = VI_SUCCESS;


  // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_PROT_FOLD_DELAY, "?") );
   viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_OUTP_FORMAT, value ) ); 


Error:
   return( error );             
}

/**************************************************************************
 *  Function: AmetekDCAttrFoldDelay_WriteCallback
 *  Purpose:  Set the foldback delay of the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrFoldDelay_WriteCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 value )
{
   ViStatus error = VI_SUCCESS;

    // Send the command.
   checkErr( AmetekDC_SendReal64Cmd(vi, io, channelName, AMETEKDC_VAL_OUTP_PROT_FOLD_DELAY, value) );

Error:
   return( error );
}


/*- AMETEKDC_ATTR_FOLDBACK_SETTING -*/
/**************************************************************************
 *  Function: AmetekDCAttrFoldDelay_ReadCallback
 *  Purpose:  Set the foldback state of the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrFoldbackSetting_WriteCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViConstString value )
{
   ViStatus    error = VI_SUCCESS;

  // Send the command.
   checkErr( AmetekDC_SendStringCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_PROT_FOLD, value) );

Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrFoldDelay_ReadCallback
 *  Purpose:  Query the foldback state of the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrFoldbackSetting_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             const ViConstString cacheValue )
{
   ViStatus    error = VI_SUCCESS;
   ViChar     mode[ SMALL_BUFF_SIZE ];


  // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_PROT_FOLD, "?") );
   viCheckErr( viScanf( io, "%s", mode ) ); 
   
   /* Now set the cache value. The IVI engine knows which channel it is for */
   viCheckErr( Ivi_SetValInStringCallback( vi, attributeId, mode ) );

Error:
   return( error );
} 

/***************************************************************************
 * ----------------- Protection Limits Callback's -------------------------*
 ***************************************************************************/

/*- AMETEKDC_ATTR_OVP_LIMIT -*/
/**************************************************************************
 *  Function: AmetekDCAttrOVPLimit_WriteCallback
 *  Purpose:  Set the OVP set point of the power supply.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOVPLimit_WriteCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViReal64 value )
{
   ViStatus error = VI_SUCCESS;

    // Send the command.
   checkErr( AmetekDC_SendReal64Cmd(vi, io, channelName, AMETEKDC_VAL_SOUR_VOLT_PROT_OVP, value) );
  
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrOVPLimit_ReadCallback
 *  Purpose:  Query the OVP set point of the power supply.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOVPLimit_ReadCallback( 
                                             ViSession vi, 
                                             ViSession io,
                                             ViConstString channelName, 
                                             ViAttr attributeId,
                                             ViReal64* value )
{
   ViStatus error = VI_SUCCESS;
   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SOUR_VOLT_PROT_OVP, "?") );
   viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_OUTP_FORMAT, value ) );
   
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrOcpLimit_ReadCallback
 *  Purpose:  Query the OCP set point of the power supply.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 *****************************************************************************/
/*- AMETEKDC_ATTR_OCP_LIMIT -*/
static ViStatus _VI_FUNC AmetekDCAttrOcpLimit_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 *value )
{
   ViStatus error = VI_SUCCESS;
   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SOUR_CURR_PROT, "?") );
   viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_INP_FORMAT, value ) );
   
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrOcpLimit_ReadCallback
 *  Purpose:  Query the OCP set point of the power supply.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOcpLimit_WriteCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 value )
{
   ViStatus error = VI_SUCCESS;

   // Since we are kludging the OCP to implement the behaviour of the current limit.
   // This function must first check to make sure that the OCP is not in tripped mode
   // because to modify the limit while in tripped mode will circuvent the intended use.
   
   // Send the command.
   checkErr( AmetekDC_SendReal64Cmd(vi, io, channelName, AMETEKDC_VAL_SOUR_CURR_PROT, value) );
   
Error:
   return( error );
}							  
				  

/*- AMETEKDC_ATTR_UVP_LIMIT -*/
/**************************************************************************
 *  Function: AmetekDCAttrUvpLimit_ReadCallback
 *  Purpose:  Query the UVP set point of the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrUvpLimit_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 *value )
{
   ViStatus error = VI_SUCCESS;
   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SOUR_VOLT_PROT_UVP, "?") );
   viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_OUTP_FORMAT, value ) );
	
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrUvpLimit_WriteCallback
 *  Purpose:  Set the UVP set point of the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrUvpLimit_WriteCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViReal64 value )
{
   ViStatus error = VI_SUCCESS;

   // Send the command.
   checkErr( AmetekDC_SendReal64Cmd(vi, io, channelName, AMETEKDC_VAL_SOUR_VOLT_PROT_UVP, value) );
	
Error:
   return( error );
}


/***************************************************************************
 * --------------------- Protection State Callbacks -----------------------*
 ***************************************************************************/
/*- AMETEKDC_ATTR_CURRENT_LIMIT_BEHAVIOR -*/
static IviRangeTableEntry attrCurrentLimitBehaviorRangeTableEntries[] =
{
   {AMETEKDC_VAL_CURRENT_REGULATE, 0, 0, "", 0},
   {AMETEKDC_VAL_CURRENT_TRIP,     0, 0, "", 1},
   {IVI_RANGE_TABLE_LAST_ENTRY}
};

static IviRangeTable attrCurrentLimitBehaviorRangeTable =
{
   IVI_VAL_DISCRETE,
   VI_FALSE,
   VI_FALSE,
   VI_NULL,
   attrCurrentLimitBehaviorRangeTableEntries,
};

/**************************************************************************
 *  Function: AmetekDCAttrCurrentLimitBehavior_ReadCallback
 *  Purpose:  Set the Current Limit behavior when the set point is exceeded.
 *
 *****************************************************************************/   
static ViStatus _VI_FUNC AmetekDCAttrCurrentLimitBehavior_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViInt32 *value )
{
   ViStatus error = VI_SUCCESS;
   ViReal64 OcpRestoreSetPoint = 0;
   ViReal64 OcpSetPoint = 0;
   
   // To kludge the behavior of the current limit I will use the OCP alarm to
   // turn off the output just like the TRIPPED behvior stipulates.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SOUR_CURR_PROT, "?") );
   viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_OUTP_FORMAT, &OcpSetPoint ) ); 
   
   // Load the stored value of current OCP set point.
   viCheckErr( AmetekDC_GetAttributeViReal64(vi, channelName, AMETEKDC_ATTR_OCP_RESTORE_VALUE, &OcpRestoreSetPoint));

   // If the current limit value 
   if( OcpSetPoint != OcpRestoreSetPoint)
   {
   	*value = AMETEKDC_VAL_CURRENT_TRIP;
   }
   else
   {
   	*value = AMETEKDC_VAL_CURRENT_REGULATE;
   }

Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrCurrentLimitBehavior_WriteCallback
 *  Purpose:  Set the Current Limit behavior when the set point is exceeded.
 *
 * NOTE: Required for IVI DC Power Supply Class Base Function.
 *
 *****************************************************************************/   
static ViStatus _VI_FUNC AmetekDCAttrCurrentLimitBehavior_WriteCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViInt32 value )
{
   ViStatus error = VI_SUCCESS;
   ViUInt32 currentBehavior = 0;
   ViReal64 OcpSetPoint = 0;
   ViReal64 currentLimit = 0;
   

   // Load the stored value of current OCP set point.
   checkErr( AmetekDC_GetAttributeViInt32(vi, channelName, AMETEKDC_ATTR_CURRENT_LIMIT_BEHAVIOR, &currentBehavior));

   // Check to see if there is a change in state by this command.
   if( currentBehavior != value)
   {
   	// This call changes the state. Execute a change.
 
 	if( value == AMETEKDC_VAL_CURRENT_TRIP)
	{
		// To kludge the behavior of the current limit I will use the OCP alarm
		// to turn off the output just like the TRIPPED behvior stipulates.
		checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SOUR_CURR_PROT, "?") );
		viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_OUTP_FORMAT, &OcpSetPoint ) ); 

		// Save the OCP setpoint restore value.
		viCheckErr( AmetekDC_SetAttributeViReal64(vi, channelName, AMETEKDC_ATTR_OCP_RESTORE_VALUE, OcpSetPoint));	

		// Get the current limit.
  	      viCheckErr( AmetekDC_GetAttributeViReal64(vi, channelName, AMETEKDC_ATTR_CURRENT_LIMIT, &currentLimit));
				
		// Send the command to set the OCP to be the same as the current limit.
		checkErr( AmetekDC_SendReal64Cmd(vi, io, channelName, AMETEKDC_VAL_SOUR_CURR_PROT, currentLimit) );
      }
	else
	{
		// Transition from tripped to regulated.
	
		// Get the OCP setpoint restore value.
		viCheckErr( AmetekDC_GetAttributeViReal64(vi, channelName, AMETEKDC_ATTR_OCP_RESTORE_VALUE, &OcpSetPoint));	
				
		// Send the command to set restore the OCP to its original set point.
		// (should be max OCP value but at some future date we might allows 
		// OCP to be used so I have coded it this way).
		checkErr( AmetekDC_SendReal64Cmd(vi, io, channelName, AMETEKDC_VAL_SOUR_CURR_PROT, OcpSetPoint) );		
	} 
  }
   
Error:
   return( error );
}

									
/*- AMETEKDC_ATTR_UVP_BEHAVIOR -*/ 
/**************************************************************************
 *  Function: AmetekDCAttrUvpBehavior_ReadCallback
 *  Purpose:  Does nothing as this value is not supported.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrUvpBehavior_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean *value )
{
   ViStatus error = VI_SUCCESS;
   
   // Command does nothing as it is not possible to regulate the UVP in XG.
   // Should generate an error or a warning to indicate the lack of functionality
   // to the user.
   *value = 0;
   
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrUvpBehavior_WriteCallback
 *  Purpose:  Does nothing as this value is not supported.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrUvpBehavior_WriteCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean value )
{
   ViStatus error = VI_SUCCESS;

   // Command does nothing as it is not possible to regulate the UVP in XG.
   // Should generate an error or a warning to indicate the lack of functionality
   // to the user.
   
Error:
   return( error );
}

/*- AMETEKDC_ATTR_CURRENT_SHARE_MODE -*/ 
/**************************************************************************
 *  Function: AmetekDCAttrCurrentShareMode_ReadCallback
 *  Purpose:  Set the current share mode of the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrCurrentShareMode_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             const ViConstString cacheValue )
{
   ViStatus error = VI_SUCCESS;
   ViChar   mode[ SMALL_BUFF_SIZE ];

   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SOUR_COMB_CSH, "?") );
   viCheckErr( viScanf( io, AMETEKDC_STRING_PARAM_INP_FORMAT, mode ) );
	
   viCheckErr( Ivi_SetValInStringCallback( vi, attributeId, mode ) );

Error:
   return( error );
}


/**************************************************************************
 *  Function: AmetekDCAttrCurrentShareMode_WriteCallback
 *  Purpose:  Set the current share mode of the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrCurrentShareMode_WriteCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViConstString value )
{
   ViStatus error = VI_SUCCESS;
   
   // Send the command.
   checkErr( AmetekDC_SendStringCmd(vi, io, channelName, AMETEKDC_VAL_SOUR_COMB_CSH, value) );

 
Error:
   return( error );
}

/*- AMETEKDC_ATTR_AC_FAIL_RESPONSE-*/ 
/**************************************************************************
 *  Function: AmetekDCAttrAcFailResponse_ReadCallback
 *  Purpose:  Query the AcFail Protection latch state.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrAcFailResponse_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean *value )
{
   ViStatus error = VI_SUCCESS;


   viCheckErr( AmetekDCGetProtectionLatch_CallbackHelper( vi, io,
                                                              channelName,
                                                              attributeId,
                                                              value,
                                                              PRT_eFLAG_ACFAIL));  
	
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrAcFailResponse_WriteCallback
 *  Purpose:  Set the AcFail Protection latch state.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrAcFailResponse_WriteCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean value )
{
   ViStatus error = VI_SUCCESS;

   viCheckErr( AmetekDCSetProtectionLatch_CallbackHelper( vi, io,
                                                              channelName,
                                                              attributeId,
                                                              value,
                                                              PRT_eFLAG_ACFAIL));   
Error:
   return( error );
}

/*- AMETEKDC_ATTR_OVER_TEMPERATURE_RESPONSE -*/ 
/**************************************************************************
 *  Function: AmetekDCAttrOverTemperatureResponse_ReadCallback
 *  Purpose:  Set the OTP Protection latch state.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOverTemperatureResponse_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean *value )
{
   ViStatus error = VI_SUCCESS;

   viCheckErr( AmetekDCGetProtectionLatch_CallbackHelper( vi, io,
                                                              channelName,
                                                              attributeId,
                                                              value,
                                                              PRT_eFLAG_OTP));   

Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrOverTemperatureResponse_WriteCallback
 *  Purpose:  Set the OTP Protection latch state.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOverTemperatureResponse_WriteCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean value )
{
   ViStatus error = VI_SUCCESS;


   viCheckErr( AmetekDCSetProtectionLatch_CallbackHelper( vi, io,
                                                              channelName,
                                                              attributeId,
                                                              value,
                                                              PRT_eFLAG_OTP));  
   	
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrFoldbackResponse_ReadCallback
 *  Purpose:  Set the Foldback Protection latch state.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrFoldbackResponse_ReadCallback (ViSession vi,
                                                                  ViSession io,
                                                                  ViConstString channelName,
                                                                  ViAttr attributeId,
                                                                  ViBoolean *value)
{
	ViStatus	error = VI_SUCCESS;


   viCheckErr( AmetekDCGetProtectionLatch_CallbackHelper( vi, io,
                                                              channelName,
                                                              attributeId,
                                                              value,
                                                              PRT_eFLAG_FLD)); 
Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrFoldbackResponse_WriteCallback
 *  Purpose:  Set the OTP Protection latch state.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrFoldbackResponse_WriteCallback (ViSession vi,
                                                                   ViSession io,
                                                                   ViConstString channelName,
                                                                   ViAttr attributeId,
                                                                   ViBoolean value)
{
	ViStatus	error = VI_SUCCESS;

   viCheckErr( AmetekDCSetProtectionLatch_CallbackHelper( vi, io,
                                                              channelName,
                                                              attributeId,
                                                              value,
                                                              PRT_eFLAG_FLD)); 
Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrOverTemperatureSlaveResponse_ReadCallback
 *  Purpose:  Set the OTP Protection latch state.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOverTemperatureSlaveResponse_ReadCallback (ViSession vi,
                                                                              ViSession io,
                                                                              ViConstString channelName,
                                                                              ViAttr attributeId,
                                                                              ViBoolean *value)
{
	ViStatus	error = VI_SUCCESS;


   viCheckErr( AmetekDCGetProtectionLatch_CallbackHelper( vi, io,
                                                              channelName,
                                                              attributeId,
                                                              value,
                                                              PRT_eFLAG_OTPSLAVE)); 
Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrOverTemperatureSlaveResponse_WriteCallback
 *  Purpose:  Set the OTP Protection latch state.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOverTemperatureSlaveResponse_WriteCallback (ViSession vi,
                                                                               ViSession io,
                                                                               ViConstString channelName,
                                                                               ViAttr attributeId,
                                                                               ViBoolean value)
{
	ViStatus	error = VI_SUCCESS;


   viCheckErr( AmetekDCSetProtectionLatch_CallbackHelper( vi, io,
                                                              channelName,
                                                              attributeId,
                                                              value,
                                                              PRT_eFLAG_OTPSLAVE)); 
Error:
	return error;
}

/*- AMETEKDC_ATTR_OUTPUT_ENABLED_POWER_ON -*/ 
/**************************************************************************
 *  Function: AmetekDCAttrOutputEnabledPowerOn_ReadCallback
 *  Purpose:  Set the Power On output state for the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOutputEnabledPowerOn_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean *value )
{
   ViStatus error = VI_SUCCESS;

   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_PON, "?") );
   viCheckErr( viScanf( io, AMETEKDC_BOOL_PARAM_INP_FORMAT, value ) ); 

Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrOutputEnabledPowerOn_WriteCallback
 *  Purpose:  Set the Power On output state for the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOutputEnabledPowerOn_WriteCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean value )
{
   ViStatus error = VI_SUCCESS;

   
   // Send the command.
   viCheckErr( AmetekDC_SendBooleanCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_PON, value) );
 
Error:
   return( error );
}


/*- AMETEKDC_ATTR_ANALOG_PROGRAMMING_RANGE -*/ 
static IviRangeTableEntry attrAnalogProgrammingRangeTableEntries[] =
	{
		{2, 10, 3, "", 0},
		{IVI_RANGE_TABLE_LAST_ENTRY}
	};

static IviRangeTable attrAnalogProgrammingRangeTable =
	{
		IVI_VAL_RANGED,
        VI_TRUE,
        VI_TRUE,
        VI_NULL,
        attrAnalogProgrammingRangeTableEntries,
	};

/**************************************************************************
 *  Function: AmetekDCAttrAnalogProgrammingRange_ReadCallback
 *  Purpose:  Set the Power On output state for the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrVoltageAnalogProgrammingRange_ReadCallback (ViSession vi,
                                                                               ViSession io,
                                                                               ViConstString channelName,
                                                                               ViAttr attributeId,
                                                                               ViReal64 *value)
{
   ViStatus error = VI_SUCCESS;

   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SYST_COMM_VAP_LEVEL, "?") );
   viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_INP_FORMAT, value ) ); 
   	
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrAnalogProgrammingRange_ReadCallback
 *  Purpose:  Set the Power On output state for the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrVoltAnalogProgrammingRange_WriteCallback (ViSession vi,
                                                                             ViSession io,
                                                                             ViConstString channelName,
                                                                             ViAttr attributeId,
                                                                             ViReal64 value)
{
   ViStatus error = VI_SUCCESS;

   // Send the command.
   checkErr( AmetekDC_SendReal64Cmd(vi, io, channelName, AMETEKDC_VAL_SYST_COMM_VAP_LEVEL, value) );

	
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrCurrAnalogProgrammingRange_ReadCallback
 *  Purpose:  Set the Power On output state for the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrCurrAnalogProgrammingRange_ReadCallback (ViSession vi,
                                                                            ViSession io,
                                                                            ViConstString channelName,
                                                                            ViAttr attributeId,
                                                                            ViReal64 *value)
{
	ViStatus	error = VI_SUCCESS;

   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SYST_COMM_CAP_LEVEL, "?") );
   viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_INP_FORMAT, value ) );
   
Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrCurrAnalogProgrammingRange_WriteCallback
 *  Purpose:  Set the Power On output state for the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrCurrAnalogProgrammingRange_WriteCallback (ViSession vi,
                                                                             ViSession io,
                                                                             ViConstString channelName,
                                                                             ViAttr attributeId,
                                                                             ViReal64 value)
{
	ViStatus	error = VI_SUCCESS;

      // Send the command.
      checkErr( AmetekDC_SendReal64Cmd(vi, io, channelName, AMETEKDC_VAL_SYST_COMM_CAP_LEVEL, value) );
   
Error:
	return error;
} 


/*- AMETEKDC_ATTR_REMOTE_MODE -*/ 
/**************************************************************************
 *  Function: AmetekDCAttrRemoteMode_ReadCallback
 *  Purpose:  Query the remote state/frontpanel lockout state for the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrRemoteMode_ReadCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             const ViConstString cacheValue )
{
   ViStatus error = VI_SUCCESS;
   ViChar   mode[ SMALL_BUFF_SIZE ];

   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SYST_REMOTE_STATE, "?") );
   viCheckErr( viScanf( io, AMETEKDC_STRING_PARAM_INP_FORMAT, mode ) ); 

   // Update the cached value.
   viCheckErr( Ivi_SetValInStringCallback( vi, attributeId, mode ) );

Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrRemoteMode_WriteCallback
 *  Purpose:  Set the remote state/frontpanel lockout state for the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrRemoteMode_WriteCallback(
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViConstString value )
{
   ViStatus error = VI_SUCCESS;

      // Send the command.
      checkErr( AmetekDC_SendStringCmd(vi, io, channelName, AMETEKDC_VAL_SYST_REMOTE_STATE, value) );
  
Error:
   return( error );
}

/*- AMETEKDC_ATTR_REMOTE_SOURCE -*/ 
/**************************************************************************
 *  Function: AmetekDCAttrRemoteSourceVolt_ReadCallback
 *  Purpose:  Query the source for setting the voltage set point for the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrRemoteSourceVolt_ReadCallback (ViSession vi,
                                                                  ViSession io,
                                                                  ViConstString channelName,
                                                                  ViAttr attributeId,
                                                                  const ViConstString cacheValue)
{
   ViStatus error = VI_SUCCESS;
   ViChar      source[ SMALL_BUFF_SIZE ];
   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SYST_REMOTE_SOUR_VOLT, "?") );
   viCheckErr( viScanf( io, AMETEKDC_STRING_PARAM_INP_FORMAT, source ) ); 
   	
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrRemoteSourceVolt_WriteCallback
 *  Purpose:  Set the source for setting the voltage set point for the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrRemoteSourceVolt_WriteCallback (ViSession vi,
                                                                   ViSession io,
                                                                   ViConstString channelName,
                                                                   ViAttr attributeId,
                                                                   ViConstString value)
{
   ViStatus error = VI_SUCCESS;

      // Send the command.
      checkErr( AmetekDC_SendStringCmd(vi, io, channelName, AMETEKDC_VAL_SYST_REMOTE_SOUR_VOLT, value) );
  
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrRemoteSourceCurr_ReadCallback
 *  Purpose:  Query the source for setting the current set point for the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrRemoteSourceCurr_ReadCallback (ViSession vi,
                                                                  ViSession io,
                                                                  ViConstString channelName,
                                                                  ViAttr attributeId,
                                                                  const ViConstString cacheValue)
{
   ViStatus error = VI_SUCCESS;
   ViChar      source[ SMALL_BUFF_SIZE ];
   
   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SYST_REMOTE_SOUR_CURR, "?") );
   viCheckErr( viScanf( io, AMETEKDC_STRING_PARAM_INP_FORMAT, source ) ); 
   	
Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCAttrRemoteSourceCurr_WriteCallback
 *  Purpose:  Set the source for setting the current set point for the power supply.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrRemoteSourceCurr_WriteCallback (ViSession vi,
                                                                   ViSession io,
                                                                   ViConstString channelName,
                                                                   ViAttr attributeId,
                                                                   ViConstString value)
{
   ViStatus error = VI_SUCCESS;


      // Send the command.
      checkErr( AmetekDC_SendStringCmd(vi, io, channelName, AMETEKDC_VAL_SYST_REMOTE_SOUR_CURR, value) );
  
Error:
   return( error );
}

/* AMETEKDC_ATTR_MODEL_TYPE */
/**************************************************************************
 *  Function: AmetekDCAttrModelType_ReadCallback
 *  Purpose:  Read the model type from the ID string.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrModelType_ReadCallback( 
                                             ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViInt32 *value )
{
   ViStatus error = VI_SUCCESS;
   ViChar IDString[ BUFFER_SIZE ];
   
   /* Ensure the session is locked before accessing this callback */
   checkErr( Ivi_GetAttributeViString( vi, channelName, 
                                       AMETEKDC_ATTR_ID_QUERY_RESPONSE,
                                       0, BUFFER_SIZE, IDString ) );

   checkErr( AmetekDC_GetModelFromBuffer( vi, IDString, value ) );
   
Error:
   return( error );
}

/* AMETEKDC_ATTR_VOLTAGE_LEVEL */
/* AMETEKDC_ATTR_TRIGGERED_VOLTAGE_LEVEL */
/* AMETEKDC_ATTR_OVP_LIMIT */
/* AMETEKDC_ATTR_UVP_LIMIT */
/* AMETEKDC_ATTR_HIGH_VOLTAGE_LIMIT */
/* AMETEKDC_ATTR_LOW_VOLTAGE_LIMIT */
/**************************************************************************
 *  Function: AmetekDCAttrVoltageLevel_RangeTableCallback
 *  Purpose:  Looks up the address of the voltage level range table that should be used.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrVoltageLevel_RangeTableCallback( 
                                             ViSession vi,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             IviRangeTablePtr *rangeTablePtr )
{
   ViStatus error = VI_SUCCESS;
   IviRangeTablePtr  tblPtr = VI_NULL;

   /* Get the range table address from the hidden attribute */
   checkErr( Ivi_GetAttributeViAddr( vi, channelName,
                                     AMETEKDC_ATTR_VOLTAGE_RANGE_TABLE_ADDRESS, 
                                     0, (void**)&tblPtr ) );

Error:
   *rangeTablePtr = tblPtr;
   return( error );
}

/* AMETEKDC_ATTR_CURRENT_LIMIT */
/* AMETEKDC_ATTR_TRIGGERED_CURRENT_LIMIT */
/* AMETEKDC_ATTR_OCP_LIMIT */
/**************************************************************************
 *  Function: AmetekDCAttrCurrentLimit_RangeTableCallback
 *  Purpose:  Looks up the address of the Current level range table that should be used.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrCurrentLimit_RangeTableCallback(
                                             ViSession vi,   
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             IviRangeTablePtr *rangeTablePtr )
{
   ViStatus error = VI_SUCCESS;
   IviRangeTablePtr  tblPtr = VI_NULL;

   /* Get the range table address from the hidden attribute */
   checkErr( Ivi_GetAttributeViAddr( vi, channelName,
                                     AMETEKDC_ATTR_CURRENT_RANGE_TABLE_ADDRESS, 
                                     0, (void**)&tblPtr ) );
   
Error:
   *rangeTablePtr = tblPtr;
   return( error );
}
								
/**************************************************************************
 *  Function: AmetekDCAttrFpInactivityTimeout_ReadCallback
 *  Purpose:  Set the front panel inactivity timeout for the power supply.
 *               Timeout is specified in seconds
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrFpInactivityTimeout_ReadCallback (ViSession vi,
                                                                     ViSession io,
                                                                     ViConstString channelName,
                                                                     ViAttr attributeId,
                                                                     ViInt32 *value)
{
	ViStatus	error = VI_SUCCESS;
	ViReal64 tmpFpTimeout = 0;

   // Send the command.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SYST_FPANEL, "?") );
   viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_INP_FORMAT, &tmpFpTimeout ) );

   // Read back value must be converted to ms.
   *value = (ViInt32)(tmpFpTimeout*1000);
Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrFpInactivityTimeout_WriteCallback
 *  Purpose:  Set the front panel inactivity timeout for the power supply.
 *               Timeout is specified in seconds
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrFpInactivityTimeout_WriteCallback (ViSession vi,
                                                                      ViSession io,
                                                                      ViConstString channelName,
                                                                      ViAttr attributeId,
                                                                      ViInt32 value)
{
	ViStatus	error = VI_SUCCESS;
	
      // Send the command.
      checkErr( AmetekDC_SendReal64Cmd(vi, io, channelName, AMETEKDC_VAL_SYST_FPANEL, ((float)value)/1000) );
   
Error:
	return error;
}

/*- AMETEKDC_ATTR_OVP_ENABLED -*/
/**************************************************************************
 *  Function: AmetekDCAttrOVPEnabled_ReadCallback
 *  Purpose:  Check the current state of the OVP. If the value is set to the 
 *                the unit max value its considered to be disabled. If not then 
 *                the protection is enabled.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOVPEnabled_ReadCallback (ViSession vi,
                                                            ViSession io,
                                                            ViConstString channelName,
                                                            ViAttr attributeId,
                                                            ViBoolean *value)
{
	ViStatus	error = VI_SUCCESS;
      ViReal64 ovpValue = 0;
      ViReal64 ovpDisabledValue = 0;

	// Get current readback value.  
      viCheckParm( Ivi_GetAttributeViReal64( vi, channelName,
                                          AMETEKDC_ATTR_OVP_DISABLE_VALUE,
                                          0, &ovpDisabledValue ), 3, "OVP Disable Value" );	
	  
      // Send the command.
      checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SOUR_VOLT_PROT_OVP, "?") );
      viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_INP_FORMAT, ovpValue ) );	

	// Compare the ovpValue to the max value.  If they are the same then 
	// ovp is enabled, else disabled.
	if( ovpDisabledValue == ovpValue)
	{
		// OVP is enabled.
		*value = 1;
	}
	else
	{
		// OVP is disabled.
		*value = 0;
	}
	
Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrOVPEnabled_WriteCallback
 *  Purpose:  Set the state of the OVP.  Since the firmware cannot disable the
 *                OVP but enable and disabling is required as part of the DC 
 *                power supply class we will simulate the disabling by setting OVP
 *                to its absolute max value.  The current setting will be stored
 *                in a local attribute which will be restored when the protection 
 *                is reenabled.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrOVPEnabled_WriteCallback (ViSession vi,
                                                             ViSession io,
                                                             ViConstString channelName,
                                                             ViAttr attributeId,
                                                             ViBoolean value)
{
	ViStatus	error = VI_SUCCESS;
      ViReal64 ovpDisabledValue = 0;
	ViBoolean ovpEnabled = VI_FALSE;

      // Load the stored value of current OVP state.
      viCheckErr( AmetekDC_GetAttributeViBoolean(vi, channelName, AMETEKDC_ATTR_OVP_ENABLED, &ovpEnabled));

	// Only make changes if there is a change in state of OVP.
	if( ovpEnabled != value )
	{
		if( value == VI_TRUE)
		{
			// Reenabled OVP, get the OVP Set point attribute
	            viCheckParm( Ivi_GetAttributeViReal64( vi, channelName,
                                          AMETEKDC_ATTR_OVP_DISABLE_VALUE,
                                          0, &ovpDisabledValue ), 4, "Get OVP Disable Value" );	
		
			// Send the disabled set point the to device to renabled.
			viCheckErr(AmetekDC_SendReal64Cmd(vi, io, channelName, AMETEKDC_VAL_SOUR_VOLT_PROT_OVP, ovpDisabledValue) );
	
		}
		else
		{
		      // Send the command.
	      		checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SOUR_VOLT_PROT_OVP, "?") );
	            viCheckErr( viScanf( io, AMETEKDC_REAL_PARAM_INP_FORMAT, &ovpDisabledValue ) );

			// Store the current OVP Set point attribute.
	            viCheckParm( Ivi_SetAttributeViReal64( vi, channelName,
                                          AMETEKDC_ATTR_OVP_DISABLE_VALUE,
                                          0, ovpDisabledValue ), 4, "Set OVP Disable Value" );				
		
		       // Read and store the current set point in the hidden OVP set point
		       // attribute.
			checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SOUR_VOLT_PROT_OVP, "MAX") );  
		}
      }
	
Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrErrorQueueCount_ReadCallback
 *  Purpose:  Read Callback function for checking the error count.
 *
 *****************************************************************************/

static ViStatus _VI_FUNC AmetekDCAttrErrorQueueCount_ReadCallback (ViSession vi, ViSession io,
                                                  ViConstString channelName,
                                                  ViAttr attributeId,
                                                  ViInt32 *value)
{
    ViStatus	error = VI_SUCCESS;

    // Send the command.
    checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SYST_ERROR_COUNT, "?") );
    viCheckErr( viScanf( io, AMETEKDC_ULONG_PARAM_INP_FORMAT, value ) );
	
Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrAuxEnabledPowerOn_ReadCallback
 *  Purpose:  Read Callback function for enabling the Aux PON settings.
 *
 *****************************************************************************/

static ViStatus _VI_FUNC AmetekDCAttrAuxEnabledPowerOn_ReadCallback (ViSession vi,
                                                                   ViSession io,
                                                                   ViConstString channelName,
                                                                   ViAttr attributeId,
                                                                   ViBoolean *value)
{

	ViStatus	error = VI_SUCCESS;

    // Send the command.
    checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_AUX_PON, "?") );
    viCheckErr( viScanf( io, AMETEKDC_BOOL_PARAM_INP_FORMAT, value ) );

Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrAuxEnabledPowerOn_WriteCallback
 *  Purpose:  Read Callback function for enabling the Aux PON settings.
 *
 *****************************************************************************/

static ViStatus _VI_FUNC AmetekDCAttrAuxEnabledPowerOn_WriteCallback (ViSession vi,
                                                                    ViSession io,
                                                                    ViConstString channelName,
                                                                    ViAttr attributeId,
                                                                    ViBoolean value)
{
	ViStatus	error = VI_SUCCESS;

    // Send the command.
    viCheckErr( AmetekDC_SendBooleanCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_AUX_PON, value) );

Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrAuxEnabled_ReadCallback
 *  Purpose:  Read Callback function for checking the Aux supply state.
 *
 *****************************************************************************/

static ViStatus _VI_FUNC AmetekDCAttrAuxEnabled_ReadCallback (ViSession vi,
                                                            ViSession io,
                                                            ViConstString channelName,
                                                            ViAttr attributeId,
                                                            ViBoolean *value)
{
	ViStatus	error = VI_SUCCESS;

    // Send the command.
    checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_AUX, "?") );
    viCheckErr( viScanf( io, AMETEKDC_BOOL_PARAM_INP_FORMAT, value ) );

Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrAuxEnabled_WriteCallback
 *  Purpose:  Read Callback function for checking the Aux supply state.
 *
 *****************************************************************************/

static ViStatus _VI_FUNC AmetekDCAttrAuxEnabled_WriteCallback (ViSession vi,
                                                             ViSession io,
                                                             ViConstString channelName,
                                                             ViAttr attributeId,
                                                             ViBoolean value)
{
	ViStatus	error = VI_SUCCESS;

    // Send the command.
    viCheckErr( AmetekDC_SendBooleanCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_AUX, value) );

Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrExtShutdownLogic_ReadCallback
 *  Purpose:  Read Callback function for checking the logic to be used for the externel shutdown pin.
 *
 *****************************************************************************/

static ViStatus _VI_FUNC AmetekDCAttrExtShutdownLogic_ReadCallback (ViSession vi,
                                                                  ViSession io,
                                                                  ViConstString channelName,
                                                                  ViAttr attributeId,
                                                                  ViBoolean *value)
{
	ViStatus	error = VI_SUCCESS;


    // Send the command.
    checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_SD_POL, "?") );
    viCheckErr( viScanf( io, AMETEKDC_BOOL_PARAM_INP_FORMAT, value ) );	

Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrExtShutdownLogic_WriteCallback
 *  Purpose:  Read Callback function for setting the logic to be used for the externel shutdown pin.
 *
 *****************************************************************************/

static ViStatus _VI_FUNC AmetekDCAttrExtShutdownLogic_WriteCallback (ViSession vi,
                                                                   ViSession io,
                                                                   ViConstString channelName,
                                                                   ViAttr attributeId,
                                                                   ViBoolean value)
{
	ViStatus	error = VI_SUCCESS;

    // Send the command.
    viCheckErr( AmetekDC_SendBooleanCmd(vi, io, channelName, AMETEKDC_VAL_OUTP_SD_POL, value) );	

Error:
	return error;
}


/**************************************************************************
 *  Function: AmetekDCAttrAutoSequenceDwellTime_ReadCallback
 *  Purpose:  Read Callback function for getting the dwell time for the auto sequence feature.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrAutoSequenceDwellTime_ReadCallback (ViSession vi,
                                                                       ViSession io,
                                                                       ViConstString channelName,
                                                                       ViAttr attributeId,
                                                                       ViInt32 *value)
{
	ViStatus	error = VI_SUCCESS;

    checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_PROG_DWELL, "?") );	
    viCheckErr( viScanf( io, AMETEKDC_ULONG_PARAM_INP_FORMAT, value ) );	
	
Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrAutoSequenceDwellTime_WriteCallback
 *  Purpose:  Write Callback function for setting the dwell time for the auto sequence feature.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrAutoSequenceDwellTime_WriteCallback (ViSession vi,
                                                                        ViSession io,
                                                                        ViConstString channelName,
                                                                        ViAttr attributeId,
                                                                        ViInt32 value)
{
	ViStatus	error = VI_SUCCESS;

    viCheckErr( AmetekDC_SendInt32Cmd(vi, io, channelName, AMETEKDC_VAL_PROG_DWELL, value) );	
	
Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrProtectionMask_ReadCallback
 *  Purpose:  Read Callback function for setting the alarm/protection masking.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrProtectionMask_ReadCallback (ViSession vi,
                                                                ViSession io,
                                                                ViConstString channelName,
                                                                ViAttr attributeId,
                                                                ViInt32 *value)
{
	ViStatus	error = VI_SUCCESS;
      checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SYST_PROT_MASK, "?") );
      viCheckErr( viScanf( io, AMETEKDC_ULONG_PARAM_INP_FORMAT, value ) );	  
	  
Error:
	return error;
}

/**************************************************************************
 *  Function: AmetekDCAttrProtectionMask_ReadCallback
 *  Purpose:  Write Callback function for setting the alarm/protection masking.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCAttrProtectionMask_WriteCallback (ViSession vi,
                                                                 ViSession io,
                                                                 ViConstString channelName,
                                                                 ViAttr attributeId,
                                                                 ViInt32 value)
{
	ViStatus	error = VI_SUCCESS;
	
      viCheckErr( AmetekDC_SendInt32Cmd(vi, io, channelName, AMETEKDC_VAL_SYST_PROT_MASK, value) );	
	  
Error:
	return error;
}

static ViStatus _VI_FUNC AmetekDCAttrConfigurableDelay_CheckCallback (ViSession vi,
                                                                    ViConstString channelName,
                                                                    ViAttr attributeId,
                                                                    ViReal64 value)
{
	ViStatus	error = VI_SUCCESS;

Error:
	return error;
}


/**************************************************************************
 *  Function: AmetekDCSetProtectionLatch_CallbackHelper
 *  Purpose:  Function for setting one of the protection latches. This 
 *               generalized function allows a central location for modifying 
 *               individual flags while reducing code space and increasing
 *               maintainability.
 *
 * Note: This is a helper function called by several attribute readback functions
 *         as such it is not part of the API.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCSetProtectionLatch_CallbackHelper(
							ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean value,
                                             ViUInt16 flagPosition)
{
   ViStatus error = VI_SUCCESS;
   ViUInt16 protLatchReg = 0; 

   // Send the command.
   // Read the latches register and mask.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SYST_PROT_LATCH, "?") );
   viCheckErr( viScanf( io, AMETEKDC_USHORT_PARAM_INP_FORMAT, &protLatchReg ) );

   // Update the latch according to the value 
   if( value == VI_FALSE)
   {
   	protLatchReg &= ~(1<<flagPosition);
   }
   else
   {
   	protLatchReg |= (1<<flagPosition);
   }

   // Send the command.
   viCheckErr( AmetekDC_SendUInt16Cmd(vi, io, channelName, AMETEKDC_VAL_SYST_PROT_LATCH, protLatchReg) );

Error:
   return( error );
}

/**************************************************************************
 *  Function: AmetekDCGetProtectionLatch_CallbackHelper
 *  Purpose:  Function for setting one of the protection latches. This 
 *               generalized function allows a central location for modifying 
 *               individual flags while reducing code space and increasing
 *               maintainability.
 *
 * Note: This is a helper function called by several attribute readback functions
 *         as such it is not part of the API.
 *
 *****************************************************************************/
static ViStatus _VI_FUNC AmetekDCGetProtectionLatch_CallbackHelper(
							ViSession vi,
                                             ViSession io,
                                             ViConstString channelName,
                                             ViAttr attributeId,
                                             ViBoolean *value,
                                             ViUInt16 flagPosition)
{
   ViStatus error = VI_SUCCESS;
   ViUInt16 protLatchReg = 0; 

   viCheckErr(Ivi_LockSession( vi, VI_NULL ));
   
   // Send the command.
   // Read the latches register and mask.
   checkErr( AmetekDC_SendCmd(vi, io, channelName, AMETEKDC_VAL_SYST_PROT_LATCH, "?") );
   viCheckErr( viScanf( io, AMETEKDC_USHORT_PARAM_INP_FORMAT, &protLatchReg ) );

    if( (protLatchReg & (1<<flagPosition)) != 0)
   {
   	*value = VI_TRUE;
   }
   else
   {
      *value = VI_FALSE;
   }
   
Error:
   Ivi_UnlockSession( vi, VI_NULL );
   return( error );
}


static ViStatus _VI_FUNC AmetekDCAttrOvpLimit_RangeTableCallback (ViSession vi,
                                                                ViConstString channelName,
                                                                ViAttr attributeId,
                                                                IviRangeTablePtr *rangeTablePtr)
{
	ViStatus	error = VI_SUCCESS;
	IviRangeTablePtr	tblPtr = VI_NULL;

   /* Get the range table address from the hidden attribute */
   checkErr( Ivi_GetAttributeViAddr( vi, channelName,
                                     AMETEKDC_ATTR_OVP_RANGE_TABLE_ADDRESS, 
                                     0, (void**)&tblPtr ) );


Error:
	*rangeTablePtr = tblPtr;
	return error;
}

/*****************************************************************************
 * Function: AmetekDC_InitAttributes                                                  
 * Purpose:  This function adds attributes to the IVI session, initializes   
 *           instrument attributes, and sets attribute invalidation          
 *           dependencies. 
 *
 *****************************************************************************/
static ViStatus AmetekDC_InitAttributes ( ViSession vi, ViInt32 modelType, ViBoolean IDQuery )
{
   ViStatus error = VI_SUCCESS;
   ViInt32  flags = 0;

   // Start indexing from this second element.
   ViInt32 idx = 1;

   ViChar supportedModelString[ BUFFER_SIZE ];
   
   ViChar idnBuffer[ BUFFER_SIZE ];
   ViString modelString;
   ViInt32 actModel = AMETEKDC_MODEL_NOT_FOUND;

   viCheckErr(Ivi_LockSession( vi, VI_NULL )); 
   
   /*- Initialize instrument attributes --------------------------------*/            
   checkErr( Ivi_SetAttributeViString( vi, VI_NULL, AMETEKDC_ATTR_GROUP_CAPABILITIES, 
                                       0, "IviDCPwrBase, "
                                       "IviDCPwrMeasurement" ) );
                                       
   checkErr( Ivi_SetAttributeViInt32( vi, VI_NULL, 
                                      AMETEKDC_ATTR_SPECIFIC_DRIVER_MAJOR_VERSION,
                                      0, AMETEKDC_MAJOR_VERSION ) );
                                       
   checkErr( Ivi_SetAttributeViInt32( vi, VI_NULL, 
                                      AMETEKDC_ATTR_SPECIFIC_DRIVER_MINOR_VERSION,
                                      0, AMETEKDC_MINOR_VERSION ) );                                      
                                       
   checkErr( Ivi_SetAttrReadCallbackViString( vi, 
                           AMETEKDC_ATTR_SPECIFIC_DRIVER_REVISION,
                           AmetekDCAttrDriverRevision_ReadCallback ) );
                                               
   checkErr( Ivi_SetAttributeViInt32( vi, "", 
                           AMETEKDC_ATTR_SPECIFIC_DRIVER_CLASS_SPEC_MAJOR_VERSION,
                           0, AMETEKDC_CLASS_SPEC_MAJOR_VERSION ) );
                                       
   checkErr( Ivi_SetAttributeViInt32( vi, "", 
                           AMETEKDC_ATTR_SPECIFIC_DRIVER_CLASS_SPEC_MINOR_VERSION,
                           0, AMETEKDC_CLASS_SPEC_MINOR_VERSION ) );
                                       
   checkErr( Ivi_SetAttributeViString( vi, "", AMETEKDC_ATTR_IO_SESSION_TYPE,
                                       0, AMETEKDC_IO_SESSION_TYPE ) );

   // Build a list of the supported models.
   sprintf( supportedModelString, "%s", AmetekDC_ModelTable[0].string);
   
   while(AmetekDC_ModelTable[idx].string != VI_NULL)
   {
   	sprintf( supportedModelString, "%s,%s", supportedModelString, AmetekDC_ModelTable[idx].string);
	idx++;
   }
   
   checkErr( Ivi_SetAttributeViString( vi, VI_NULL, AMETEKDC_ATTR_SUPPORTED_INSTRUMENT_MODELS, 
                                       IVI_VAL_NOT_USER_WRITABLE , supportedModelString ) );
                                        
   checkErr( Ivi_SetAttrReadCallbackViString( vi, 
                           AMETEKDC_ATTR_INSTRUMENT_FIRMWARE_REVISION,
                           AmetekDCAttrInstrumentFirmwareRevision_ReadCallback ) );
                                               
   checkErr( Ivi_GetAttributeFlags( vi, 
                                    AMETEKDC_ATTR_INSTRUMENT_MANUFACTURER, 
                                    &flags ) );
     
   checkErr( Ivi_SetAttributeFlags( vi, 
                                    AMETEKDC_ATTR_INSTRUMENT_MANUFACTURER, 
                                    flags | 
                                    IVI_VAL_USE_CALLBACKS_FOR_SIMULATION ) );
                                     
   checkErr( Ivi_SetAttrReadCallbackViString( vi, 
                           AMETEKDC_ATTR_INSTRUMENT_MANUFACTURER,
                           AmetekDCAttrInstrumentManufacturer_ReadCallback ) );
                                               
   checkErr( Ivi_GetAttributeFlags( vi, AMETEKDC_ATTR_INSTRUMENT_MODEL, 
                                    &flags ) );
     
   checkErr( Ivi_SetAttributeFlags( vi, AMETEKDC_ATTR_INSTRUMENT_MODEL, 
                                    flags | 
                                    IVI_VAL_USE_CALLBACKS_FOR_SIMULATION ) );
                                     
   checkErr( Ivi_SetAttrReadCallbackViString( vi, 
                           AMETEKDC_ATTR_INSTRUMENT_MODEL,
                           AmetekDCAttrInstrumentModel_ReadCallback ) );
                                               
   checkErr( Ivi_SetAttributeViString( vi, "", 
									   AMETEKDC_ATTR_SPECIFIC_DRIVER_VENDOR,
                                       0, AMETEKDC_DRIVER_VENDOR ) );
                                        
   checkErr( Ivi_SetAttributeViString( vi, "", 
                                       AMETEKDC_ATTR_SPECIFIC_DRIVER_DESCRIPTION,
                                       0, AMETEKDC_DRIVER_DESCRIPTION ) );
                                        
   checkErr( Ivi_SetAttributeViAddr( vi, VI_NULL, 
                                     IVI_ATTR_CHECK_STATUS_CALLBACK, 0,
                                     AmetekDC_CheckStatusCallback ) );
                                      
   checkErr( Ivi_SetAttributeViBoolean( vi, VI_NULL, 
                                        IVI_ATTR_SUPPORTS_WR_BUF_OPER_MODE, 
                                        0, VI_TRUE ) );
                                         
   /*- Instrument-specific attributes ------------------------------*/     
   
   // The following three attributes are used in the detecting and setup
   // of the model so they are added here.  Do not change the order in which
   // these attributes are added.
	checkErr (Ivi_AddAttributeViReal64 (vi, AMETEKDC_ATTR_CONFIGURABLE_DELAY,
	                                    "AMETEKDC_ATTR_CONFIGURABLE_DELAY", 0.03, 0,
	                                    VI_NULL, VI_NULL,
	                                    &attrConfigurableDelayRangeTable, 0));
	                                    
	checkErr (Ivi_SetAttrCheckCallbackViReal64 (vi, AMETEKDC_ATTR_CONFIGURABLE_DELAY,
	                                            AmetekDCAttrConfigurableDelay_CheckCallback));

   // Create the response string to the IDN query for the virtual machine.
   Ivi_GetStringFromTable( AmetekDC_ModelTable, AMETEKDC_DEFAULT_MODEL_TYPE, &modelString );
   sprintf( idnBuffer, AMETEKDC_DEFAULT_DEF_ID_STRING,  modelString); 
   checkErr( Ivi_AddAttributeViString( vi, AMETEKDC_ATTR_ID_QUERY_RESPONSE,
                           "AMETEKDC_ATTR_ID_QUERY_RESPONSE", idnBuffer,
                           IVI_VAL_NOT_USER_WRITABLE | IVI_VAL_MULTI_CHANNEL,
                           AmetekDCAttrIdQueryResponse_ReadCallback,
                           VI_NULL ) );

   // Setup the model.
   if( !Ivi_Simulating( vi ) )
   {
      ViSession io = Ivi_IOSession( vi );
	  
      /* Query the ID if IDQuery chosen and the model is still unknown */
      if( IDQuery )
      {
         AmetekDCQueryID_CallbackHelper( vi, io, AMETEKDC_MASTER_CHANNEL, BUFFER_SIZE, idnBuffer );
         AmetekDC_GetModelFromBuffer( vi, idnBuffer, &actModel );
		 
         if(( modelType != actModel ) && (modelType != AMETEKDC_MODEL_NOT_FOUND ))
         {
            viCheckErr( VI_ERROR_FAIL_ID_QUERY );
         } 
         else
         {
         	modelType = actModel;
         }
      }    	        	  
   }
                           
   if( modelType == AMETEKDC_MODEL_NOT_FOUND )
   {
      modelType = AMETEKDC_DEFAULT_MODEL_TYPE;
      
      Ivi_GetStringFromTable( AmetekDC_ModelTable, modelType, &modelString );

      // Create the response string to the IDN query for the virtual machine.
   	  sprintf( idnBuffer, AMETEKDC_DEFAULT_DEF_ID_STRING, modelString );
   }
	  
   //Set to new value based on what was detected by the IDN query.
   checkErr( Ivi_SetAttributeViString( vi, AMETEKDC_MASTER_CHANNEL, 
                                           AMETEKDC_ATTR_ID_QUERY_RESPONSE, 
                                           IVI_VAL_DONT_MARK_AS_SET_BY_USER,
                                           idnBuffer ) );	  
                                             
   
	checkErr (Ivi_AddAttributeViBoolean (vi, AMETEKDC_ATTR_OUTPUT_ENABLED,
	                                     "AMETEKDC_ATTR_OUTPUT_ENABLED", VI_FALSE,
	                                     IVI_VAL_MULTI_CHANNEL | IVI_VAL_NEVER_CACHE,
	                                     AmetekDCAttrOutputEnabled_ReadCallback,
	                                     AmetekDCAttrOutputEnabled_WriteCallback));

                                      
                                        
                                 
   checkErr( Ivi_AddAttributeViReal64( vi, AMETEKDC_ATTR_VOLTAGE_LEVEL,
                           "AMETEKDC_ATTR_VOLTAGE_LEVEL", 0.0,
                           IVI_VAL_MULTI_CHANNEL,
                           AmetekDCAttrVoltageLevel_ReadCallback,
                           AmetekDCAttrVoltageLevel_WriteCallback,
                           VI_NULL, 4 ) );
                           
   checkErr( Ivi_SetAttrRangeTableCallback( vi, AMETEKDC_ATTR_VOLTAGE_LEVEL,
                           AmetekDCAttrVoltageLevel_RangeTableCallback ) );
    
	checkErr (Ivi_AddAttributeViReal64 (vi, AMETEKDC_ATTR_OVP_LIMIT,
	                                    "AMETEKDC_ATTR_OVP_LIMIT", 7.5,
	                                    IVI_VAL_MULTI_CHANNEL,
	                                    AmetekDCAttrOVPLimit_ReadCallback,
	                                    AmetekDCAttrOVPLimit_WriteCallback, VI_NULL, 4));
	checkErr (Ivi_SetAttrRangeTableCallback (vi, AMETEKDC_ATTR_OVP_LIMIT,
	                                         AmetekDCAttrOvpLimit_RangeTableCallback));
                           
   checkErr( Ivi_SetAttrRangeTableCallback( vi, AMETEKDC_ATTR_OVP_LIMIT,
                           AmetekDCAttrOvpLimit_RangeTableCallback ) );
   
   checkErr( Ivi_AddAttributeViReal64( vi, AMETEKDC_ATTR_CURRENT_LIMIT,
                           "AMETEKDC_ATTR_CURRENT_LIMIT", 0.0,
                           IVI_VAL_MULTI_CHANNEL,
                           AmetekDCAttrCurrentLimit_ReadCallback,
                           AmetekDCAttrCurrentLimit_WriteCallback,
                           VI_NULL, 4 ) );
                           
   checkErr( Ivi_SetAttrRangeTableCallback( vi, AMETEKDC_ATTR_CURRENT_LIMIT,
                           AmetekDCAttrCurrentLimit_RangeTableCallback )  );


   checkErr( Ivi_AddAttributeViInt32( vi, AMETEKDC_ATTR_OPC_TIMEOUT,
                           "AMETEKDC_ATTR_OPC_TIMEOUT", 1000,
                           IVI_VAL_DONT_CHECK_STATUS | IVI_VAL_HIDDEN,
                           VI_NULL, VI_NULL, VI_NULL ) );
                           
	checkErr (Ivi_AddAttributeViReal64 (vi, AMETEKDC_ATTR_FOLD_DELAY,
	                                    "AMETEKDC_ATTR_FOLD_DELAY", 
	                                    AMETEKDC_FOLDBACK_MIN_DELAY_SEC,
	                                    IVI_VAL_MULTI_CHANNEL,
	                                    AmetekDCAttrFoldDelay_ReadCallback,
	                                    AmetekDCAttrFoldDelay_WriteCallback,
	                                    &attrFoldDelayRangeTable, 0));
    
   checkErr( Ivi_AddAttributeViString( vi, AMETEKDC_ATTR_FOLDBACK_SETTING,
                           "AMETEKDC_ATTR_FOLDBACK_SETTING", "NONE",
	                                    IVI_VAL_MULTI_CHANNEL | IVI_VAL_NEVER_CACHE,
                           AmetekDCAttrFoldbackSetting_ReadCallback,
                           AmetekDCAttrFoldbackSetting_WriteCallback ) );
                           
                           
                                       
   checkErr( Ivi_AddAttributeViReal64( vi, AMETEKDC_ATTR_OCP_LIMIT,
                           "AMETEKDC_ATTR_OCP_LIMIT", 0.0,
                           IVI_VAL_MULTI_CHANNEL,
                           AmetekDCAttrOcpLimit_ReadCallback,
                           AmetekDCAttrOcpLimit_WriteCallback,
                           VI_NULL, 4 ) );
                           
   checkErr (Ivi_SetAttrRangeTableCallback( vi, AMETEKDC_ATTR_OCP_LIMIT,
                           AmetekDCAttrCurrentLimit_RangeTableCallback ) );
   
                                       
   checkErr( Ivi_AddAttributeViReal64( vi, AMETEKDC_ATTR_UVP_LIMIT,
                           "AMETEKDC_ATTR_UVP_LIMIT", 0.0,
                           IVI_VAL_MULTI_CHANNEL,
                           AmetekDCAttrUvpLimit_ReadCallback,
                           AmetekDCAttrUvpLimit_WriteCallback,
                           VI_NULL, 4 ) );
                           
   checkErr( Ivi_SetAttrRangeTableCallback( vi, AMETEKDC_ATTR_UVP_LIMIT,
                           AmetekDCAttrVoltageLevel_RangeTableCallback ) );
                                       
                                      
   checkErr( Ivi_AddAttributeViInt32( vi, AMETEKDC_ATTR_CURRENT_LIMIT_BEHAVIOR,
                           "AMETEKDC_ATTR_CURRENT_LIMIT_BEHAVIOR",
                           AMETEKDC_VAL_CURRENT_REGULATE,
                           IVI_VAL_MULTI_CHANNEL,
                           AmetekDCAttrCurrentLimitBehavior_ReadCallback,
                           AmetekDCAttrCurrentLimitBehavior_WriteCallback,
                           &attrCurrentLimitBehaviorRangeTable ) );
                           
                                        
                                        
                                        
   checkErr( Ivi_AddAttributeViBoolean( vi, AMETEKDC_ATTR_UVP_BEHAVIOR,
                           "AMETEKDC_ATTR_UVP_BEHAVIOR",
                           AMETEKDC_VAL_PROTECTION_WARNING,
                           IVI_VAL_MULTI_CHANNEL,
                           AmetekDCAttrUvpBehavior_ReadCallback,
                           AmetekDCAttrUvpBehavior_WriteCallback ) );
                                        
	checkErr (Ivi_AddAttributeViString (vi, AMETEKDC_ATTR_CURRENT_SHARE_MODE,
	                                    "AMETEKDC_ATTR_CURRENT_SHARE_MODE", "MAST",
	                                    IVI_VAL_MULTI_CHANNEL | IVI_VAL_NEVER_CACHE,
	                                    AmetekDCAttrCurrentShareMode_ReadCallback,
	                                    AmetekDCAttrCurrentShareMode_WriteCallback));
                           
   checkErr( Ivi_AddAttributeViBoolean( vi, 
                           AMETEKDC_ATTR_OVER_TEMPERATURE_RESPONSE,
                           "AMETEKDC_ATTR_OVER_TEMPERATURE_RESPONSE",
                           VI_TRUE, IVI_VAL_MULTI_CHANNEL,
                           AmetekDCAttrOverTemperatureResponse_ReadCallback,
                           AmetekDCAttrOverTemperatureResponse_WriteCallback ) );
                                        
   checkErr( Ivi_AddAttributeViBoolean( vi, AMETEKDC_ATTR_AC_FAIL_RESPONSE,
                           "AMETEKDC_ATTR_AC_FAIL_RESPONSE", VI_FALSE,
                           IVI_VAL_MULTI_CHANNEL,
                           AmetekDCAttrAcFailResponse_ReadCallback,
                           AmetekDCAttrAcFailResponse_WriteCallback ) );
                           
                                       
                                       
   checkErr(Ivi_AddAttributeViBoolean( vi, AMETEKDC_ATTR_OUTPUT_ENABLED_POWER_ON,
                           "AMETEKDC_ATTR_OUTPUT_ENABLED_POWER_ON",
                           VI_FALSE, IVI_VAL_MULTI_CHANNEL,
                           AmetekDCAttrOutputEnabledPowerOn_ReadCallback,
                           AmetekDCAttrOutputEnabledPowerOn_WriteCallback ) );
                                        
                                       
   checkErr(Ivi_AddAttributeViString( vi, AMETEKDC_ATTR_REMOTE_MODE,
                           "AMETEKDC_ATTR_REMOTE_MODE", "LOC",
	                                    IVI_VAL_MULTI_CHANNEL | IVI_VAL_NEVER_CACHE,
                           AmetekDCAttrRemoteMode_ReadCallback,
                           AmetekDCAttrRemoteMode_WriteCallback ) );
                                       
	checkErr (Ivi_AddAttributeViReal64 (vi,
	                                    AMETEKDC_ATTR_VOLT_ANALOG_PROGRAMMING_RANGE,
	                                    "AMETEKDC_ATTR_VOLT_ANALOG_PROGRAMMING_RANGE",
	                                    10.0, IVI_VAL_MULTI_CHANNEL,
	                                    AmetekDCAttrVoltageAnalogProgrammingRange_ReadCallback,
	                                    AmetekDCAttrVoltAnalogProgrammingRange_WriteCallback,
	                                    &attrAnalogProgrammingRangeTable, 0));
                                      
                                      
                                      
	checkErr (Ivi_AddAttributeViString (vi, AMETEKDC_ATTR_REMOTE_SOURCE_VOLT,
	                                    "AMETEKDC_ATTR_REMOTE_SOURCE_VOLT", "LOC",
	                                    IVI_VAL_MULTI_CHANNEL | IVI_VAL_NEVER_CACHE,
	                                    AmetekDCAttrRemoteSourceVolt_ReadCallback,
	                                    AmetekDCAttrRemoteSourceVolt_WriteCallback));
                                       
                           
   checkErr( Ivi_AddAttributeViInt32( vi, AMETEKDC_ATTR_MODEL_TYPE,
                           "AMETEKDC_ATTR_MODEL_TYPE", AMETEKDC_DEFAULT_MODEL_TYPE,
                           IVI_VAL_NOT_USER_WRITABLE | IVI_VAL_MULTI_CHANNEL | 
                           IVI_VAL_USE_CALLBACKS_FOR_SIMULATION,
                           AmetekDCAttrModelType_ReadCallback, VI_NULL,
                           VI_NULL ) );
                           
   checkErr( Ivi_AddAttributeViAddr( vi, 
                           AMETEKDC_ATTR_VOLTAGE_RANGE_TABLE_ADDRESS,
                           "AMETEKDC_ATTR_VOLTAGE_RANGE_TABLE_ADDRESS",
                           AmetekDC_kazModelOutputRangeTable[AMETEKDC_DEFAULT_MODEL_TYPE].addrVoltRangeTbl ,
                           IVI_VAL_MULTI_CHANNEL | IVI_VAL_HIDDEN,
                           VI_NULL, VI_NULL ) );
                           
   checkErr( Ivi_AddAttributeViAddr( vi, 
                           AMETEKDC_ATTR_CURRENT_RANGE_TABLE_ADDRESS,
                           "AMETEKDC_ATTR_CURRENT_RANGE_TABLE_ADDRESS",
                           AmetekDC_kazModelOutputRangeTable[AMETEKDC_DEFAULT_MODEL_TYPE].addrCurrRangeTbl,
                           IVI_VAL_MULTI_CHANNEL | IVI_VAL_HIDDEN,
                           VI_NULL, VI_NULL ) );
                           
                           
	checkErr (Ivi_AddAttributeViString (vi, AMETEKDC_ATTR_MULTICHANNEL_MODEL,
	                                    "AMETEKDC_ATTR_MULTICHANNEL_MODEL",
	                                    AMETEKDC_DEFAULT_MODEL_STRING,
	                                    IVI_VAL_MULTI_CHANNEL | IVI_VAL_USE_CALLBACKS_FOR_SIMULATION | IVI_VAL_HIDDEN,
	                                    AmetekDCAttrMultichannelModel_ReadCallback,
	                                    VI_NULL));
                           
   checkErr( Ivi_AddAttributeViString( vi, 
                           AMETEKDC_ATTR_MULTICHANNEL_FIRMWARE_REVISION,
                           "AMETEKDC_ATTR_MULTICHANNEL_FIRMWARE_REVISION",
                           AMETEKDC_DEFAULT_ID_STRING,
                           IVI_VAL_NOT_USER_WRITABLE | IVI_VAL_MULTI_CHANNEL | 
                           IVI_VAL_USE_CALLBACKS_FOR_SIMULATION,
                           AmetekDCAttrMultichannelFirmwareRevision_ReadCallback,
                           VI_NULL ) );                           
	
	checkErr (Ivi_AddAttributeViBoolean (vi,
	                                     AMETEKDC_ATTR_OVER_TEMPERATURE_SLAVE_RESPONSE,
	                                     "AMETEKDC_ATTR_OVER_TEMPERATURE_SLAVE_RESPONSE",
	                                     VI_TRUE, IVI_VAL_MULTI_CHANNEL,
	                                     AmetekDCAttrOverTemperatureSlaveResponse_ReadCallback,
	                                     AmetekDCAttrOverTemperatureSlaveResponse_WriteCallback));
	
	checkErr (Ivi_AddAttributeViBoolean (vi, AMETEKDC_ATTR_FOLDBACK_RESPONSE,
	                                     "AMETEKDC_ATTR_FOLDBACK_RESPONSE", VI_TRUE,
	                                     IVI_VAL_MULTI_CHANNEL,
	                                     AmetekDCAttrFoldbackResponse_ReadCallback,
	                                     AmetekDCAttrFoldbackResponse_WriteCallback));
	
	checkErr (Ivi_AddAttributeViReal64 (vi,
	                                    AMETEKDC_ATTR_CURR_ANALOG_PROGRAMMING_RANGE,
	                                    "AMETEKDC_ATTR_CURR_ANALOG_PROGRAMMING_RANGE",
	                                    10, IVI_VAL_MULTI_CHANNEL,
	                                    AmetekDCAttrCurrAnalogProgrammingRange_ReadCallback,
	                                    AmetekDCAttrCurrAnalogProgrammingRange_WriteCallback,
	                                    &attrAnalogProgrammingRangeTable, 0));
	
	checkErr (Ivi_AddAttributeViString (vi, AMETEKDC_ATTR_REMOTE_SOURCE_CURR,
	                                    "AMETEKDC_ATTR_REMOTE_SOURCE_CURR", "LOC",
	                                    IVI_VAL_MULTI_CHANNEL | IVI_VAL_NEVER_CACHE,
	                                    AmetekDCAttrRemoteSourceCurr_ReadCallback,
	                                    AmetekDCAttrRemoteSourceCurr_WriteCallback));
	
	checkErr (Ivi_AddAttributeViInt32 (vi, AMETEKDC_ATTR_FP_INACTIVITY_TIMEOUT,
	                                   "AMETEKDC_ATTR_FP_INACTIVITY_TIMEOUT", 3000,
	                                   IVI_VAL_MULTI_CHANNEL,
	                                   AmetekDCAttrFpInactivityTimeout_ReadCallback,
	                                   AmetekDCAttrFpInactivityTimeout_WriteCallback,
	                                   &attrFpInactivityTimeoutRangeTable));
	
	checkErr (Ivi_AddAttributeViBoolean (vi, AMETEKDC_ATTR_OVP_ENABLED,
	                                     "AMETEKDC_ATTR_OVP_ENABLED", VI_TRUE,
	                                     IVI_VAL_MULTI_CHANNEL,
	                                     AmetekDCAttrOVPEnabled_ReadCallback,
	                                     AmetekDCAttrOVPEnabled_WriteCallback));
	
	checkErr (Ivi_AddAttributeViReal64 (vi, AMETEKDC_ATTR_OVP_DISABLE_VALUE,
	                                    "AMETEKDC_ATTR_OVP_DISABLE_VALUE", 0,
	                                    IVI_VAL_ALWAYS_CACHE | IVI_VAL_HIDDEN,
	                                    VI_NULL, VI_NULL, VI_NULL, 0));
	
	checkErr (Ivi_AddAttributeViReal64 (vi, AMETEKDC_ATTR_OCP_RESTORE_VALUE,
	                                    "AMETEKDC_ATTR_OCP_RESTORE_VALUE", 0,
	                                    IVI_VAL_MULTI_CHANNEL | IVI_VAL_ALWAYS_CACHE | IVI_VAL_HIDDEN,
	                                    VI_NULL, VI_NULL, VI_NULL, 0));
	
	checkErr (Ivi_AddAttributeViBoolean (vi, AMETEKDC_ATTR_AUX_ENABLED,
	                                     "AMETEKDC_ATTR_AUX_ENABLED", VI_FALSE,
	                                     IVI_VAL_MULTI_CHANNEL | IVI_VAL_NEVER_CACHE,
	                                     AmetekDCAttrAuxEnabled_ReadCallback,
	                                     AmetekDCAttrAuxEnabled_WriteCallback));
	
	checkErr (Ivi_AddAttributeViBoolean (vi, AMETEKDC_ATTR_AUX_ENABLED_POWER_ON,
	                                     "AMETEKDC_ATTR_AUX_ENABLED_POWER_ON",
	                                     VI_FALSE, IVI_VAL_MULTI_CHANNEL,
	                                     AmetekDCAttrAuxEnabledPowerOn_ReadCallback,
	                                     AmetekDCAttrAuxEnabledPowerOn_WriteCallback));
	
	checkErr (Ivi_AddAttributeViBoolean (vi, AMETEKDC_ATTR_EXT_SHUTDOWN_LOGIC,
	                                     "AMETEKDC_ATTR_EXT_SHUTDOWN_LOGIC", 0,
	                                     IVI_VAL_MULTI_CHANNEL,
	                                     AmetekDCAttrExtShutdownLogic_ReadCallback,
	                                     AmetekDCAttrExtShutdownLogic_WriteCallback));
	checkErr (Ivi_AddAttributeViInt32 (vi, AMETEKDC_ATTR_AUTO_SEQUENCE_DWELL_TIME,
	                                   "AMETEKDC_ATTR_AUTO_SEQUENCE_DWELL_TIME", 0,
	                                   IVI_VAL_MULTI_CHANNEL,
	                                   AmetekDCAttrAutoSequenceDwellTime_ReadCallback,
	                                   AmetekDCAttrAutoSequenceDwellTime_WriteCallback,
	                                   &attrAutoSequenceDwellTimeRangeTable));
	checkErr (Ivi_AddAttributeViInt32 (vi, AMETEKDC_ATTR_PROTECTION_MASK,
	                                   "AMETEKDC_ATTR_PROTECTION_MASK", 1023,
	                                   IVI_VAL_MULTI_CHANNEL | IVI_VAL_NEVER_CACHE,
	                                   AmetekDCAttrProtectionMask_ReadCallback,
	                                   AmetekDCAttrProtectionMask_WriteCallback,
	                                   VI_NULL));
	checkErr (Ivi_AddAttributeViAddr (vi, AMETEKDC_ATTR_OVP_RANGE_TABLE_ADDRESS,
	                                  "AMETEKDC_ATTR_OVP_RANGE_TABLE_ADDRESS",
	                                  AmetekDC_kazModelOutputRangeTable[AMETEKDC_DEFAULT_MODEL_TYPE].addrOvpRangeTbl,
	                                  IVI_VAL_MULTI_CHANNEL | IVI_VAL_HIDDEN,
	                                  VI_NULL, VI_NULL));

Error:
    Ivi_UnlockSession( vi, VI_NULL ); 
    return( error );
}

/*****************************************************************************
 *------------------- End Instrument Driver Source Code ---------------------*
 *****************************************************************************/
